/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * builder_init.cpp
 *
 *  Created on: Dec 15, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <iomanip>
#include <iostream>

#include "Actions/ActionHistory.hpp"
#include "Actions/ActionExceptions.hpp"

#include "builder_init.hpp"
#include "cleanUp.hpp"

#include "CodePatterns/Log.hpp"

#include "Graph/BondGraph.hpp"

#include "Parser/ChangeTracker.hpp"
#include "Parser/FormatParserStorage.hpp"

#include "UIElements/UIFactory.hpp"
#include "UIElements/CommandLineUI/CommandLineParser.hpp"
#include "UIElements/CommandLineUI/CommandLineUIFactory.hpp"
#ifdef USE_GUI_QT
#include "UIElements/Qt4/QtUIFactory.hpp"
#endif
#include "UIElements/TextUI/TextUIFactory.hpp"
#include "UIElements/MainWindow.hpp"

#include "version.h"

#include "World.hpp"


/** Print some initial information output the program.
 *
 */
void ProgramHeader()
{
  // print version check and copyright notice
  cout << MOLECUILDERVERSION << endl;
  cout << "MoleCuilder comes with ABSOLUTELY NO WARRANTY; for details type" << endl;
  cout << "`molecuilder --warranty'." << endl;
  cout << "`MoleCuilder - to create and alter molecular systems." << endl;
  cout << "Copyright (C) 2010  University Bonn. All rights reserved." << endl;
  cout << endl;
}

/** General stuff to intialize before UI.
 *
 */
void initGeneral()
{
  // while we are non interactive, we want to abort from asserts
  ASSERT_DO(Assert::Abort);
  ASSERT_HOOK(dumpMemory);

  ProgramHeader();

  setVerbosity(1);
  // need to init the history before any action is created
  MoleCuilder::ActionHistory::init();

  // from this moment on, we need to be sure to deeinitialize in the correct order
  // this is handled by the cleanup function
  atexit(cleanUp);
}

/** Initialize specific UIFactory.
 *
 * @param argc argument count
 * @param argv argument array
 */
void initUI(int argc, char **argv)
{
  std::string BondGraphFileName("\n");
  // Parse command line options and if present create respective UI
  // construct bond graph
  if (boost::filesystem::exists(BondGraphFileName)) {
    std::ifstream input(BondGraphFileName.c_str());
    if ((input.good()) && (World::getInstance().getBondGraph()->LoadBondLengthTable(input))) {
      LOG(0, "Bond length table loaded successfully.");
    } else {
      ELOG(1, "Bond length table loading failed.");
    }
    input.close();
  }
  // handle remaining arguments by CommandLineParser
  if (argc>1) {
    LOG(0, "Setting UI to CommandLine.");
    CommandLineParser::getInstance().InitializeCommandArguments();
    CommandLineParser::getInstance().Run(argc,argv);
    UIFactory::registerFactory(new CommandLineUIFactory::description());
    UIFactory::makeUserInterface("CommandLine");
  } else {
    // In the interactive mode, we can leave the user the choice in case of error
    ASSERT_DO(Assert::Ask);
    #ifdef USE_GUI_QT
      LOG(0, "Setting UI to Qt4.");
      UIFactory::registerFactory(new QtUIFactory::description());
      UIFactory::makeUserInterface("Qt4");
    #else
      LOG(0, "Setting UI to Text.");
      cout << MOLECUILDERVERSION << endl;
      UIFactory::registerFactory(new TextUIFactory::description());
      UIFactory::makeUserInterface("Text");
    #endif
  }
}

/** Create MainWindow and displays.
 * I.e. here all the Actions are parsed and done.
 */
void doUI()
{
  MainWindow *mainWindow = UIFactory::getInstance().makeMainWindow();
  try {
    mainWindow->display();
  } catch(ActionFailureException &e) {
    std::cerr << "Action " << *boost::get_error_info<ActionNameString>(e) << " has failed." << std::endl;
    World::getInstance().setExitFlag(5);
  }
  delete mainWindow;
}

/** Dump current memory chunks.
 *
 */
void dumpMemory()
{
  ofstream ost("molecuilder.memdump");
  Memory::dumpMemory(ost);
}

/** Save the current World to output files and exit.
 *
 * @return retrieved from World::getExitFlag()
 */
int saveAll()
{
  FormatParserStorage::getInstance().SaveAll();
  ChangeTracker::getInstance().saveStatus();

  int ExitFlag = World::getInstance().getExitFlag();
  return (ExitFlag == 1 ? 0 : ExitFlag);
}
