/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * AddEmptyBoundaryAction.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
#include "boost/serialization/vector.hpp"

#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "Box.hpp"
#include "CodePatterns/Log.hpp"
#include "LinearAlgebra/MatrixContent.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "World.hpp"

#include <iostream>
#include <string>
#include <vector>

#include "Actions/WorldAction/AddEmptyBoundaryAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "AddEmptyBoundaryAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr WorldAddEmptyBoundaryAction::performCall() {
  Vector Min;
  Vector Max;
  int j=0;

  // obtain information
  getParametersfromValueStorage();

  // create undo domain
  std::stringstream undostream;
  boost::archive::text_oarchive oa(undostream);
  const RealSpaceMatrix &matrix = World::getInstance().getDomain().getM();
  oa << matrix;

  // get maximum and minimum
  std::vector<atom *> AllAtoms = World::getInstance().getAllAtoms();
  if (AllAtoms.size() > 0) {
    std::vector<atom *>::iterator AtomRunner = AllAtoms.begin();
    Min = (*AtomRunner)->getPosition();
    Max = (*AtomRunner)->getPosition();
    for (; AtomRunner != AllAtoms.end(); ++AtomRunner) {
      for (int i=0;i<NDIM;i++) {
        if ((*AtomRunner)->at(i) > Max[i])
          Max[i] = (*AtomRunner)->at(i);
        if ((*AtomRunner)->at(i) < Min[i])
          Min[i] = (*AtomRunner)->at(i);
      }
    }
    // set new box size
    double * const cell_size = new double[6];
    for (j=0;j<6;j++)
      cell_size[j] = 0.;
    j=-1;
    for (int i=0;i<NDIM;i++) {
      j += i+1;
      cell_size[j] = (Max[i]-Min[i]+2.*params.boundary[i]);
    }
    World::getInstance().setDomain(cell_size);
    delete[] cell_size;
    // translate all atoms, such that Min is aty (0,0,0)
    AtomRunner = AllAtoms.begin();
    for (std::vector<atom *>::iterator AtomRunner = AllAtoms.begin();
        AtomRunner != AllAtoms.end();
        ++AtomRunner)
      *(*AtomRunner) -= Min - params.boundary;

    // give final box size
    LOG(0, "Box domain is now " << World::getInstance().getDomain().getM());

    // create undo state
    WorldAddEmptyBoundaryState *UndoState =
        new WorldAddEmptyBoundaryState(
            undostream.str(),
            World::getInstance().getDomain().getM(),
            Min,
            params
            );

    return Action::state_ptr(UndoState);
  } else {
    ELOG(2, "There must be atoms present for AddingEmptyBoundary.");
    return Action::failure;
  }
}

Action::state_ptr WorldAddEmptyBoundaryAction::performUndo(Action::state_ptr _state) {
  WorldAddEmptyBoundaryState *state = assert_cast<WorldAddEmptyBoundaryState*>(_state.get());

  // restore domain
  RealSpaceMatrix matrix;
  std::stringstream undostream(state->undostring);
  boost::archive::text_iarchive ia(undostream);
  ia >> matrix;
  World::getInstance().setDomain(matrix);

  // give final box size
  LOG(0, "Box domain restored to " << World::getInstance().getDomain().getM());

  // restore atoms
  std::vector<atom *> AllAtoms = World::getInstance().getAllAtoms();
  for (std::vector<atom *>::iterator AtomRunner = AllAtoms.begin();
      AtomRunner != AllAtoms.end();
      ++AtomRunner)
    *(*AtomRunner) += state->Min - state->params.boundary;

  return Action::state_ptr(_state);
}

Action::state_ptr WorldAddEmptyBoundaryAction::performRedo(Action::state_ptr _state){
  WorldAddEmptyBoundaryState *state = assert_cast<WorldAddEmptyBoundaryState*>(_state.get());

  World::getInstance().setDomain(state->newdomain);

  // give final box size
  LOG(0, "Box domain is again " << World::getInstance().getDomain().getM());

  // shift atoms
  std::vector<atom *> AllAtoms = World::getInstance().getAllAtoms();
  for (std::vector<atom *>::iterator AtomRunner = AllAtoms.begin();
      AtomRunner != AllAtoms.end();
      ++AtomRunner)
    *(*AtomRunner) -= state->Min - state->params.boundary;

  return Action::state_ptr(_state);
}

bool WorldAddEmptyBoundaryAction::canUndo() {
  return true;
}

bool WorldAddEmptyBoundaryAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
