/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ShapeOpsUnitTest.cpp
 *
 *  Created on: Jun 18, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <cmath>

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

#include "CodePatterns/Assert.hpp"

#include "Helpers/defs.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "Shapes/BaseShapes.hpp"
#include "Shapes/Shape.hpp"
#include "Shapes/ShapeOps.hpp"

#include "ShapeOpsUnitTest.hpp"

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ShapeOpsTest );

void ShapeOpsTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);
}

void ShapeOpsTest::tearDown()
{}

void ShapeOpsTest::resizeTest()
{
  Shape s = resize(Sphere(), 2.);

  CPPUNIT_ASSERT( s.isInside( Vector(1.5,0.,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(0.,1.5,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(0.,0.,1.5) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(2.,0.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,2.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,0.,2.) ) );
}

void ShapeOpsTest::translateTest()
{
  Shape s = translate(Sphere(), Vector(1.,0.,0.));

  CPPUNIT_ASSERT( s.isInside( Vector(1.5,0.,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(1.,.5,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(1.,0.,.5) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(2.,0.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(1.,1.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(1.,0.,1.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(-.5,0.,0.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(0.,-.5,0.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(0.,0.,-.5) ) );
}

void ShapeOpsTest::stretchTest()
{
#ifndef NDEBUG
  CPPUNIT_ASSERT_THROW( Shape test = stretch(Sphere(), Vector(-2.,0.,0.)), Assert::AssertionFailure );
#endif
  Shape s = stretch(Sphere(), Vector(2.,1.,1.));

  CPPUNIT_ASSERT( s.isInside( Vector(1.5,0.,0.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(0.,1.5,0.) ) );
  CPPUNIT_ASSERT( !s.isInside( Vector(0.,0.,1.5) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(2.,0.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,1.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,1.,0.) ) );
}

void ShapeOpsTest::transformTest()
{
  RealSpaceMatrix M;
  M.setRotation(45., 0., 0.);
  Shape s = transform(Sphere(), M);

  CPPUNIT_ASSERT( s.isInside( Vector(.5,0.,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(0.,.5,0.) ) );
  CPPUNIT_ASSERT( s.isInside( Vector(0.,0.,.5) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(1.,0.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,1.,0.) ) );
  CPPUNIT_ASSERT( s.isOnSurface( Vector(0.,0.,1.) ) );
}

