/*
 * CommandLineWindow.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

#include <boost/bind.hpp>

#include "CommandLineUI/CommandLineWindow.hpp"
#include "CommandLineUI/CommandLineStatusIndicator.hpp"

#include "Actions/ActionRegistry.hpp"
#include "Actions/AnalysisAction/MolecularVolumeAction.hpp"
#include "Actions/AnalysisAction/PairCorrelationAction.hpp"
#include "Actions/AnalysisAction/PrincipalAxisSystemAction.hpp"
#include "Actions/AtomAction/AddAction.hpp"
#include "Actions/AtomAction/ChangeElementAction.hpp"
#include "Actions/AtomAction/RemoveAction.hpp"
#include "Actions/CmdAction/BondLengthTableAction.hpp"
#include "Actions/CmdAction/ElementDbAction.hpp"
#include "Actions/CmdAction/FastParsingAction.hpp"
#include "Actions/CmdAction/HelpAction.hpp"
#include "Actions/CmdAction/VerboseAction.hpp"
#include "Actions/CmdAction/VersionAction.hpp"
#include "Actions/FragmentationAction/DepthFirstSearchAction.hpp"
#include "Actions/FragmentationAction/SubgraphDissectionAction.hpp"
#include "Actions/FragmentationAction/FragmentationAction.hpp"
#include "Actions/MoleculeAction/BondFileAction.hpp"
#include "Actions/MoleculeAction/ChangeNameAction.hpp"
#include "Actions/MoleculeAction/FillWithMoleculeAction.hpp"
#include "Actions/MoleculeAction/LinearInterpolationofTrajectoriesAction.hpp"
#include "Actions/MoleculeAction/SaveAdjacencyAction.hpp"
#include "Actions/MoleculeAction/SaveBondsAction.hpp"
#include "Actions/MoleculeAction/SaveTemperatureAction.hpp"
#include "Actions/MoleculeAction/VerletIntegrationAction.hpp"
#include "Actions/ParserAction/LoadXyzAction.hpp"
#include "Actions/ParserAction/SaveXyzAction.hpp"
#include "Actions/TesselationAction/ConvexEnvelopeAction.hpp"
#include "Actions/TesselationAction/NonConvexEnvelopeAction.hpp"
#include "Actions/WorldAction/AddEmptyBoundaryAction.hpp"
#include "Actions/WorldAction/BoundInBoxAction.hpp"
#include "Actions/WorldAction/CenterInBoxAction.hpp"
#include "Actions/WorldAction/CenterOnEdgeAction.hpp"
#include "Actions/WorldAction/ChangeBoxAction.hpp"
#include "Actions/WorldAction/RemoveSphereOfAtomsAction.hpp"
#include "Actions/WorldAction/RepeatBoxAction.hpp"
#include "Actions/WorldAction/ScaleBoxAction.hpp"
#include "Actions/WorldAction/SetDefaultNameAction.hpp"
#include "Actions/WorldAction/SetGaussianBasisAction.hpp"
#include "CommandLineParser.hpp"

#include <iostream>

using namespace std;

// TODO: see what code can be moved to a base class for Graphic and CommandLine Windows
CommandLineWindow::CommandLineWindow()
{
  // create and register all command line callable actions
  populateAnalysisActions();
  populateAtomActions();
  populateCmdActions();
  populateFragmentationActions();
  populateMoleculeActions();
  populateParserActions();
  populateTesselationActions();
  populateWorldActions();

  // Add status indicators etc...
  statusIndicator = new CommandLineStatusIndicator();
}

CommandLineWindow::~CommandLineWindow()
{
  delete statusIndicator;
}

void CommandLineWindow::display() {
  // go through all possible actions
  for (std::list<std::string>::iterator CommandRunner = CommandLineParser::getInstance().SequenceOfActions.begin(); CommandRunner != CommandLineParser::getInstance().SequenceOfActions.end(); ++CommandRunner) {
    cout << "Checking presence of " << *CommandRunner << endl;
    if (ActionRegistry::getInstance().isActionByNamePresent(*CommandRunner))
      ActionRegistry::getInstance().getActionByName(*CommandRunner)->call();
  }
}

void CommandLineWindow::populateAnalysisActions()
{
  new AnalysisMolecularVolumeAction();
  new AnalysisPairCorrelationAction();
  new AnalysisPrincipalAxisSystemAction();
}

void CommandLineWindow::populateAtomActions()
{
  new AtomAddAction();
  new AtomChangeElementAction();
  new AtomRemoveAction();
}

void CommandLineWindow::populateCmdActions()
{
  new CommandLineBondLengthTableAction();
  new CommandLineElementDbAction();
  new CommandLineFastParsingAction();
  new CommandLineHelpAction();
  new CommandLineVerboseAction();
  new CommandLineVersionAction();
}

void CommandLineWindow::populateFragmentationActions()
{
  new FragmentationDepthFirstSearchAction();
  new FragmentationFragmentationAction();
  new FragmentationSubgraphDissectionAction();
}

void CommandLineWindow::populateMoleculeActions()
{
  new MoleculeBondFileAction();
  new MoleculeChangeNameAction();
  new MoleculeFillWithMoleculeAction();
  new MoleculeLinearInterpolationofTrajectoriesAction();
  new MoleculeSaveAdjacencyAction();
  new MoleculeSaveBondsAction();
  new MoleculeSaveTemperatureAction();
  new MoleculeVerletIntegrationAction();
}

void CommandLineWindow::populateParserActions()
{
  new ParserLoadXyzAction();
  new ParserSaveXyzAction();
}

void CommandLineWindow::populateTesselationActions()
{
  new TesselationConvexEnvelopeAction();
  new TesselationNonConvexEnvelopeAction();
}

void CommandLineWindow::populateWorldActions()
{
  new WorldAddEmptyBoundaryAction();
  new WorldBoundInBoxAction();
  new WorldCenterInBoxAction();
  new WorldCenterOnEdgeAction();
  new WorldChangeBoxAction();
  new WorldRemoveSphereOfAtomsAction();
  new WorldRepeatBoxAction();
  new WorldScaleBoxAction();
  new WorldSetDefaultNameAction();
  new WorldSetGaussianBasisAction();
}
