/*
 * SaturationPotential.hpp
 *
 *  Created on: Oct 11, 2012
 *      Author: heber
 */

#ifndef SATURATIONPOTENTIAL_HPP_
#define SATURATIONPOTENTIAL_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/function.hpp>
#include <iosfwd>
#include <limits>

#include "Potentials/EmpiricalPotential.hpp"
#include "Potentials/SerializablePotential.hpp"
#include "Potentials/Specifics/PairPotential_Angle.hpp"
#include "Potentials/Specifics/PairPotential_Morse.hpp"
#include "FunctionApproximation/FunctionModel.hpp"

/** This is a combination of Morse and Angle potentials for saturated elements.
 *
 */
class SaturationPotential :
  virtual public EmpiricalPotential,
  virtual public FunctionModel,
  virtual public SerializablePotential
{
  //!> grant unit test access to internal parts
  friend class SaturationPotentialTest;
  //!> grant operator access to private functions
  friend std::ostream& operator<<(std::ostream &ost, const SaturationPotential &potential);
  //!> grant operator access to private functions
  friend std::istream& operator>>(std::istream &ost, SaturationPotential &potential);

  // some repeated typedefs to avoid ambiguities
  typedef FunctionModel::arguments_t arguments_t;
  typedef FunctionModel::result_t result_t;
  typedef FunctionModel::results_t results_t;
  typedef EmpiricalPotential::derivative_components_t derivative_components_t;
  typedef FunctionModel::parameters_t parameters_t;
public:
  SaturationPotential(
      const ParticleTypes_t &_ParticleTypes,
      const double _saturation_cutoff,
      boost::function< std::vector<arguments_t>(const argument_t &, const double)> &_triplefunction
      );
  SaturationPotential(
      const ParticleTypes_t &_ParticleTypes,
      const double _all_energy_offset,
      const double _morse_spring_constant,
      const double _morse_equilibrium_distance,
      const double _morse_dissociation_energy,
      const double _angle_spring_constant,
      const double _angle_equilibrium_distance,
      const double _saturation_cutoff,
      boost::function< std::vector<arguments_t>(const argument_t &, const double)> &_triplefunction
      );
  virtual ~SaturationPotential() {}

  /** Setter for parameters as required by FunctionModel interface.
   *
   * \param _params given set of parameters
   */
  void setParameters(const parameters_t &_params);

  /** Getter for parameters as required by FunctionModel interface.
   *
   * \return set of parameters
   */
  parameters_t getParameters() const;

  /** Getter for the number of parameters of this model function.
   *
   * \return number of parameters
   */
  size_t getParameterDimension() const
  { return MAXPARAMS; }

  /** Evaluates the harmonic potential function for the given arguments.
   *
   * @param arguments single distance
   * @return value of the potential function
   */
  results_t operator()(const arguments_t &arguments) const;

  /** Evaluates the derivative of the potential function.
   *
   * @param arguments single distance
   * @return vector with derivative with respect to the input degrees of freedom
   */
  derivative_components_t derivative(const arguments_t &arguments) const;

  /** Evaluates the derivative of the function with the given \a arguments
   * with respect to a specific parameter indicated by \a index.
   *
   * \param arguments set of arguments as input variables to the function
   * \param index derivative of which parameter
   * \return result vector containing the derivative with respect to the given
   *         input
   */
  results_t parameter_derivative(const arguments_t &arguments, const size_t index) const;

  /** Return the token name of this specific potential.
   *
   * \return token name of the potential
   */
  const std::string& getToken() const
  { return potential_token; }

  /** Returns a vector of parameter names.
   *
   * This is required from the specific implementation
   *
   * \return vector of strings containing parameter names
   */
  const ParameterNames_t& getParameterNames() const
  { return ParameterNames; }

  /** States whether lower and upper boundaries should be used to constraint
   * the parameter search for this function model.
   *
   * \return true - constraints should be used, false - else
   */
  bool isBoxConstraint() const {
    return true;
  }

  /** Returns a vector which are the lower boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling lowest allowed values
   */
  parameters_t getLowerBoxConstraints() const {
    parameters_t lowerbounds(getParameterDimension(), -std::numeric_limits<double>::max());
    lowerbounds[morse_equilibrium_distance] = 0.;
    lowerbounds[angle_equilibrium_distance] = -1.;
    return lowerbounds;
  }

  /** Returns a vector which are the upper boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling highest allowed values
   */
  parameters_t getUpperBoxConstraints() const {
    parameters_t upperbounds(getParameterDimension(), std::numeric_limits<double>::max());
    upperbounds[angle_equilibrium_distance] = 1.;
    return upperbounds;
  }

  /** Returns a bound function to be used with TrainingData, extracting distances
   * from a Fragment.
   *
   * \param charges vector of charges to be extracted
   * \return bound function extracting distances from a fragment
   */
  FunctionModel::extractor_t getFragmentSpecificExtractor(const charges_t &charges) const;

  enum parameter_enum_t {
    all_energy_offset,
    morse_spring_constant,
    morse_equilibrium_distance,
    morse_dissociation_energy,
    angle_spring_constant,
    angle_equilibrium_distance,
    MAXPARAMS
  };

private:
  /** Adds last bond partner one more time to get types for angle potential.
   *
   * @param _ParticleTypes types from a pair potential
   * @return types for a symmetric triple potential
   */
  static const ParticleTypes_t symmetrizeTypes(const ParticleTypes_t &_ParticleTypes);

private:
  PairPotential_Morse morse;
  PairPotential_Angle angle;
  double energy_offset;

  //!> bound function that obtains the triples for the internal coordinationb summation.
  const boost::function< std::vector< arguments_t >(const argument_t &, const double)> &triplefunction;
  const double saturation_cutoff;

  //!> static definitions of the parameter name for this potential
  static const ParameterNames_t ParameterNames;

  //!> static token of this potential type
  static const std::string potential_token;
};

/** Output operations stores .potentials line containing these parameters
 * coefficients and designations.
 *
 * \note We have to overrride this function in order to print both potentials
 * separately.
 *
 * \param ost output stream to print to
 * \param potential potential whose coefficients to print
 * \return output stream for concatenation
 */
std::ostream& operator<<(std::ostream &ost, const SaturationPotential &potential);

/** Input operation parses coefficients from a given line of a .potentials
 * file.
 *
 * \note We have to override this function in order to parse both potentials
 * separately.
 *
 * \param ist input stream to parse from
 * \param potential potential to set
 * \return input stream for concatenation
 */
std::istream& operator>>(std::istream &ist, SaturationPotential &potential);

#endif /* SATURATIONPOTENTIAL_HPP_ */
