/*
 * PairPotential_Angle.hpp
 *
 *  Created on: Oct 11, 2012
 *      Author: heber
 */

#ifndef PAIRPOTENTIAL_ANGLE_HPP_
#define PAIRPOTENTIAL_ANGLE_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <limits>

#include "CodePatterns/Assert.hpp"

#include "Potentials/EmpiricalPotential.hpp"
#include "Potentials/SerializablePotential.hpp"
#include "FunctionApproximation/FunctionModel.hpp"

/** This is the implementation of a harmonic angle potential.
 *
 * This evaluates \f$ k \cdot (\theta -\theta_0)^2 \f$.
 *
 */
class PairPotential_Angle :
  virtual public EmpiricalPotential,
  virtual public FunctionModel,
  virtual public SerializablePotential
{
  //!> grant unit test access to internal parts
  friend class PairPotential_AngleTest;
  // some repeated typedefs to avoid ambiguities
  typedef FunctionModel::arguments_t arguments_t;
  typedef FunctionModel::result_t result_t;
  typedef FunctionModel::results_t results_t;
  typedef EmpiricalPotential::derivative_components_t derivative_components_t;
  typedef FunctionModel::parameters_t parameters_t;
public:
  PairPotential_Angle(const ParticleTypes_t &_ParticleTypes);
  PairPotential_Angle(
      const ParticleTypes_t &_ParticleTypes,
      const double _spring_constant,
      const double _equilibrium_distance,
      const double _energy_offset);
  virtual ~PairPotential_Angle() {}

  /** Setter for parameters as required by FunctionModel interface.
   *
   * \param _params given set of parameters
   */
  void setParameters(const parameters_t &_params);

  /** Getter for parameters as required by FunctionModel interface.
   *
   * \return set of parameters
   */
  parameters_t getParameters() const
  {
    return params;
  }

  /** Getter for the number of parameters of this model function.
   *
   * \return number of parameters
   */
  size_t getParameterDimension() const
  {
    return 3;
  }

  /** Evaluates the harmonic potential function for the given arguments.
   *
   * @param arguments single distance
   * @return value of the potential function
   */
  results_t operator()(const arguments_t &arguments) const;

  /** Evaluates the derivative of the potential function.
   *
   * @param arguments single distance
   * @return vector with derivative with respect to the input degrees of freedom
   */
  derivative_components_t derivative(const arguments_t &arguments) const;

  /** Evaluates the derivative of the function with the given \a arguments
   * with respect to a specific parameter indicated by \a index.
   *
   * \param arguments set of arguments as input variables to the function
   * \param index derivative of which parameter
   * \return result vector containing the derivative with respect to the given
   *         input
   */
  results_t parameter_derivative(const arguments_t &arguments, const size_t index) const;

  /** Return the token name of this specific potential.
   *
   * \return token name of the potential
   */
  const std::string& getToken() const
  { return potential_token; }

  /** Returns a vector of parameter names.
   *
   * This is required from the specific implementation
   *
   * \return vector of strings containing parameter names
   */
  const ParameterNames_t& getParameterNames() const
  { return ParameterNames; }

  /** States whether lower and upper boundaries should be used to constraint
   * the parameter search for this function model.
   *
   * \return true - constraints should be used, false - else
   */
  bool isBoxConstraint() const {
    return true;
  }

  /** Returns a vector which are the lower boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling lowest allowed values
   */
  parameters_t getLowerBoxConstraints() const {
    parameters_t lowerbounds(getParameterDimension(), -std::numeric_limits<double>::max());
    lowerbounds[equilibrium_distance] = -1.;
    return lowerbounds;
  }

  /** Returns a vector which are the upper boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling highest allowed values
   */
  parameters_t getUpperBoxConstraints() const {
    parameters_t upperbounds(getParameterDimension(), std::numeric_limits<double>::max());
    upperbounds[equilibrium_distance] = 1.;
    return upperbounds;
  }

  /** Returns a bound function to be used with TrainingData, extracting distances
   * from a Fragment.
   *
   * \param charges vector of charges to be extracted
   * \return bound function extracting distances from a fragment
   */
  FunctionModel::extractor_t getFragmentSpecificExtractor(const charges_t &charges) const;

  enum parameter_enum_t {
    spring_constant=0,
    equilibrium_distance=1,
    energy_offset=2,
    MAXPARAMS
  };

private:
  result_t
  function_theta(
      const double &r_ij,
      const double &r_ik,
      const double &r_jk
    ) const;
private:
  //!> parameter vector with parameters as in enum parameter_enum_t
  parameters_t params;

  //!> static definitions of the parameter name for this potential
  static const ParameterNames_t ParameterNames;

  //!> static token of this potential type
  static const std::string potential_token;
};

#endif /* PAIRPOTENTIAL_ANGLE_HPP_ */
