/*
 * AtomFragmentsMap.hpp
 *
 *  Created on: Mar 7, 2016
 *      Author: heber
 */


#ifndef ATOMFRAGMENTSMAP_HPP_
#define ATOMFRAGMENTSMAP_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/Singleton.hpp"

#include <list>
#include <map>
#include <vector>

#include <boost/serialization/export.hpp>
#include <boost/serialization/list.hpp>
#include <boost/serialization/map.hpp>
#include <boost/serialization/set.hpp>
#include <boost/serialization/vector.hpp>
#include <boost/serialization/version.hpp>

#include "types.hpp"

class KeySet;
class Graph;

/** This class creates in instantiation a map connecting each atom with
 * the (known) fragments it takes part in.
 *
 * In the HomologyGraph and its -Container we do not have this information
 * any longer. However, we need this in order to make statements about atomic
 * properties from calculated fragment properties.
 *
 */
class AtomFragmentsMap : public Singleton<AtomFragmentsMap>
{
public:
  //** Function to insert new fragments into storage container.
  void insert(
      const Graph &_graph);

  /** Function to clear the container.
   *
   */
  void clear()
  { atommap.clear(); }

  typedef std::vector<atomId_t> indices_t;
  typedef std::list<KeySet> keysets_t;
  //!> typedef for the internal map
  typedef std::map<atomId_t, keysets_t> AtomFragmentsMap_t;
  typedef std::map<KeySet, indices_t > FragmentFullKeysetMap_t;

  /** Getter for full stored map.
   *
   * \return const ref to internal map
   */
  const AtomFragmentsMap_t& getMap() const
  { return atommap; }

  /** Allows to add the full keyset, with excluded hydrogens, to add
   * to a given \a _keyset
   *
   * \param _keyset keyset to a fragment without hydrogens
   * \param _fullkeyset full keyset with excluded hydrogens to associate with \a _keyset
   * \return true - insertion ok, else - index set already present
   */
  bool addFullKeyset(const KeySet &_keyset, const indices_t &_fullkeyset);

  /** Getter for the full key set, i.e. including excluded hydrogens, for a
   * given \a _keyset without them.
   *
   * \param _keyset keyset to a fragment without hydrogens
   * \return full index set containing all keys from \a _keyset and all excluded
   *         hydrogens
   */
  const indices_t &getFullKeyset(const KeySet &_keyset) const;

  /** Getter to map cut down to given selection of atoms.
   *
   * \param _candidates subset of atoms
   * \param _MaxOrder constrain returned fragment list to contain at most this size
   * \return map with fragments for each of the candidates
   */
  AtomFragmentsMap_t getMap(
      const std::vector<atomId_t> &_candidates,
      size_t _MaxOrder) const;

  /** Checks whether we have a full keyset for every keyset contained.
   *
   * \return true - is complete, false - else
   */
  bool checkCompleteness() const;

private:
  //!> grant singleton pattern access to private cstor/dstor
  friend class Singleton<AtomFragmentsMap>;

  /** Private default cstor.
   *
   */
  AtomFragmentsMap() {}

  /** Private default dstor.
   *
   */
  ~AtomFragmentsMap() {}

private:
  //!> internal map associating atoms and fragments
  AtomFragmentsMap_t atommap;

  //!> internal map to get from keyset (without hydrogens) to full keyset, i.e. forcekeyset
  FragmentFullKeysetMap_t fullkeysets;

private:
  friend class boost::serialization::access;
  // serialization
  template <typename Archive>
  void serialize(Archive& ar, const unsigned int version)
  {
    ar & atommap;
    ar & fullkeysets;
  }
};


#endif /* ATOMFRAGMENTSMAP_HPP_ */
