/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * AsyncOperation.cpp
 *
 *  Created on: Nov 11, 2011
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <boost/bind.hpp>
#include <vector>
#include "Connection.hpp" // Must come before boost/serialization headers.
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"

#include "Controller/Commands/AsyncOperation.hpp"

/** Callback function when an operation has been completed.
 *
 * \param e error code if something went wrong
 */
void AsyncOperation::handle_FinishOperation(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    LOG(1, "INFO: AsyncOperation completed.");
  }
  else
  {
    // An error occurred.
    Exitflag = ErrorFlag;
    ELOG(1, e.message());
  }

  // Since we are not starting a new operation the io_service will run out of
  // work to do and the client will exit.
  disconnect();
}

/** Internal function to disconnect connection_ correctly.
 *
 */
void AsyncOperation::disconnect()
{
  connection_.socket().close();
}

void AsyncOperation::operator()(const std::string& _host, const std::string& _service)
{
  Info info(__FUNCTION__);
  // Resolve the host name into an IP address.
  boost::asio::ip::tcp::resolver resolver(connection_.socket().get_io_service());
  boost::asio::ip::tcp::resolver::query query(_host, _service);
  boost::asio::ip::tcp::resolver::iterator endpoint_iterator =
    resolver.resolve(query);
  boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;

  // Start an asynchronous connect operation.
  std::cout << "Connecting to endpoint " << endpoint << " ..." << std::endl;
  connection_.socket().async_connect(endpoint,
    boost::bind(&AsyncOperation::handle_connect, this,
      boost::asio::placeholders::error, ++endpoint_iterator));
}

