/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * TextDialog.cpp
 *
 *  Created on: Jan 5, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include <iostream>

#include <Descriptors/AtomDescriptor.hpp>
#include <Descriptors/AtomIdDescriptor.hpp>
#include <Descriptors/MoleculeDescriptor.hpp>
#include <Descriptors/MoleculeIdDescriptor.hpp>
#include <boost/lexical_cast.hpp>
#include "TextUI/TextDialog.hpp"

#include "World.hpp"
#include "periodentafel.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"

#include "atom.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "LinearAlgebra/Matrix.hpp"
#include "Box.hpp"

#include <boost/lexical_cast.hpp>
#include <boost/filesystem.hpp>

using namespace std;

using boost::lexical_cast;
using boost::bad_lexical_cast;


TextDialog::TextDialog()
{
}

TextDialog::~TextDialog()
{
}

void TextDialog::queryEmpty(const char* title, std::string description){
  registerQuery(new EmptyTextQuery(title,description));
}

void TextDialog::queryBoolean(const char* title, std::string description){
  registerQuery(new BooleanTextQuery(title,description));
}

void TextDialog::queryInt(const char* title, std::string description){
  registerQuery(new IntTextQuery(title,description));
}

void TextDialog::queryInts(const char* title, std::string description){
  registerQuery(new IntsTextQuery(title,description));
}

void TextDialog::queryDouble(const char* title, std::string description){
  registerQuery(new DoubleTextQuery(title,description));
}

void TextDialog::queryDoubles(const char* title, std::string description){
  registerQuery(new DoublesTextQuery(title,description));
}

void TextDialog::queryString(const char* title, std::string description){
  registerQuery(new StringTextQuery(title,description));
}

void TextDialog::queryStrings(const char* title, std::string description){
  registerQuery(new StringsTextQuery(title,description));
}

void TextDialog::queryAtom(const char* title, std::string description) {
  registerQuery(new AtomTextQuery(title,description));
}

void TextDialog::queryAtoms(const char* title, std::string description) {
  registerQuery(new AtomsTextQuery(title,description));
}

void TextDialog::queryMolecule(const char* title, std::string description) {
  registerQuery(new MoleculeTextQuery(title,description));
}

void TextDialog::queryMolecules(const char* title, std::string description) {
  registerQuery(new MoleculesTextQuery(title,description));
}

void TextDialog::queryVector(const char* title, bool check, std::string description) {
  registerQuery(new VectorTextQuery(title,check,description));
}

void TextDialog::queryVectors(const char* title, bool check, std::string description) {
  registerQuery(new VectorsTextQuery(title,check,description));
}

void TextDialog::queryBox(const char* title, std::string description) {
  registerQuery(new BoxTextQuery(title,description));
}

void TextDialog::queryElement(const char* title, std::string description){
  registerQuery(new ElementTextQuery(title,description));
}

void TextDialog::queryElements(const char* title, std::string description){
  registerQuery(new ElementsTextQuery(title,description));
}

void TextDialog::queryFile(const char* title, std::string description){
  registerQuery(new FileTextQuery(title,description));
}

/************************** Query Infrastructure ************************/

TextDialog::EmptyTextQuery::EmptyTextQuery(string title, std::string _description) :
    Dialog::EmptyQuery(title,_description)
{}

TextDialog::EmptyTextQuery::~EmptyTextQuery() {}

bool TextDialog::EmptyTextQuery::handle() {
  cout << "Message of " << getTitle() << ":\n" << getDescription() << "\n";
  return true;
}

TextDialog::IntTextQuery::IntTextQuery(string title, std::string _description) :
    Dialog::IntQuery(title,_description)
{}

TextDialog::IntTextQuery::~IntTextQuery() {}

bool TextDialog::IntTextQuery::handle() {
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> tmp;
    if(cin.fail()){
      badInput=true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
    }
  } while(badInput);
  // clear the input buffer of anything still in the line
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return true;
}

TextDialog::IntsTextQuery::IntsTextQuery(string title, std::string _description) :
    Dialog::IntsQuery(title,_description)
{}

TextDialog::IntsTextQuery::~IntsTextQuery() {}

bool TextDialog::IntsTextQuery::handle() {
  Log() << Verbose(0) << getTitle();
  std::string line;
  getline(cin,line);
  // dissect by " "
  std::string::iterator olditer = line.begin();
  for(string::iterator iter = line.begin(); iter != line.end(); ++iter) {
    if (*iter == ' ') {
      std::istringstream stream(string(iter, olditer));
      stream >> temp;
      tmp.push_back(temp);
      olditer = iter;
    }
  }
  if (olditer != line.begin()) { // insert last part also
    std::istringstream stream(string(olditer, line.end()));
    stream >> temp;
    tmp.push_back(temp);
  }

  return true;
}

TextDialog::BooleanTextQuery::BooleanTextQuery(string title, std::string _description) :
    Dialog::BooleanQuery(title,_description)
{}

TextDialog::BooleanTextQuery::~BooleanTextQuery() {}

bool TextDialog::BooleanTextQuery::handle() {
  bool badInput = false;
  char input = ' ';
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> input;
    if ((input == 'y' ) || (input == 'Y')) {
      tmp = true;
    } else if ((input == 'n' ) || (input == 'N')) {
      tmp = false;
    } else {
      badInput=true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not of [yYnN]!" << endl;
    }
  } while(badInput);
  // clear the input buffer of anything still in the line
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return true;
}

TextDialog::StringTextQuery::StringTextQuery(string title, std::string _description) :
    Dialog::StringQuery(title,_description)
{}

TextDialog::StringTextQuery::~StringTextQuery() {}

bool TextDialog::StringTextQuery::handle() {
  Log() << Verbose(0) << getTitle();
  getline(cin,tmp);
  return true;
}

TextDialog::StringsTextQuery::StringsTextQuery(string title, std::string _description) :
    Dialog::StringsQuery(title,_description)
{}

TextDialog::StringsTextQuery::~StringsTextQuery() {}

bool TextDialog::StringsTextQuery::handle() {
  Log() << Verbose(0) << getTitle();
  getline(cin,temp);
  // dissect by " "
  std::string::iterator olditer = temp.begin();
  for(string::iterator iter = temp.begin(); iter != temp.end(); ++iter) {
    if (*iter == ' ') {
      tmp.push_back(string(iter, olditer));
      olditer = iter;
    }
  }
  if (olditer != temp.begin())  // insert last part also
    tmp.push_back(string(olditer, temp.end()));

  return true;
}

TextDialog::DoubleTextQuery::DoubleTextQuery(string title, std::string _description) :
    Dialog::DoubleQuery(title,_description)
{}

TextDialog::DoubleTextQuery::~DoubleTextQuery() {}

bool TextDialog::DoubleTextQuery::handle() {
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> tmp;
    if(cin.fail()){
      badInput = true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
    }
  }while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return true;
}


TextDialog::DoublesTextQuery::DoublesTextQuery(string title, std::string _description) :
    Dialog::DoublesQuery(title,_description)
{}

TextDialog::DoublesTextQuery::~DoublesTextQuery() {}

bool TextDialog::DoublesTextQuery::handle() {
  Log() << Verbose(0) << getTitle();
  std::string line;
  getline(cin,line);
  // dissect by " "
  std::string::iterator olditer = line.begin();
  for(string::iterator iter = line.begin(); iter != line.end(); ++iter) {
    if (*iter == ' ') {
      std::istringstream stream(string(iter, olditer));
      stream >> temp;
      tmp.push_back(temp);
      olditer = iter;
    }
  }
  if (olditer != line.begin()) { // insert last part also
    std::istringstream stream(string(olditer, line.end()));
    stream >> temp;
    tmp.push_back(temp);
  }

  return true;
}

TextDialog::AtomTextQuery::AtomTextQuery(string title, std::string _description) :
    Dialog::AtomQuery(title,_description)
{}

TextDialog::AtomTextQuery::~AtomTextQuery() {}

bool TextDialog::AtomTextQuery::handle() {
  int idxOfAtom=-1;
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> idxOfAtom;
    if(cin.fail()){
      badInput = true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
      continue;
    }

    tmp = World::getInstance().getAtom(AtomById(idxOfAtom));
    if(!tmp && idxOfAtom!=-1){
      Log() << Verbose(0) << "Invalid Atom Index" << idxOfAtom << endl;
      badInput = true;
    }

  } while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return (idxOfAtom!=-1);
}


TextDialog::AtomsTextQuery::AtomsTextQuery(string title, std::string _description) :
    Dialog::AtomsQuery(title,_description)
{}

TextDialog::AtomsTextQuery::~AtomsTextQuery() {}

bool TextDialog::AtomsTextQuery::handle() {
  int idxOfAtom=-1;
  Log() << Verbose(0) << getTitle();
  std::string line;
  getline(cin,line);
  // dissect by " "
  std::string::iterator olditer = line.begin();
  for(string::iterator iter = line.begin(); iter != line.end(); ++iter) {
    if (*iter == ' ') {
      std::istringstream stream(string(iter, olditer));
      stream >> idxOfAtom;
      temp = World::getInstance().getAtom(AtomById(idxOfAtom));
      if(!temp && idxOfAtom!=-1){
        Log() << Verbose(0) << "Invalid Atom Index" << idxOfAtom << endl;
        break;
      }
      tmp.push_back(temp);
      olditer = iter;
    }
  }
  if (olditer != line.begin()) { // insert last part also
    std::istringstream stream(string(olditer, line.end()));
    stream >> idxOfAtom;
    temp = World::getInstance().getAtom(AtomById(idxOfAtom));
    if(!temp && idxOfAtom!=-1) {
      Log() << Verbose(0) << "Invalid Atom Index" << idxOfAtom << endl;
      tmp.push_back(temp);
    }
  }

  return (idxOfAtom!=-1);
}

TextDialog::MoleculeTextQuery::MoleculeTextQuery(string title, std::string _description) :
    Dialog::MoleculeQuery(title,_description)
{}

TextDialog::MoleculeTextQuery::~MoleculeTextQuery() {}

bool TextDialog::MoleculeTextQuery::handle() {
  int idxOfMol=0;
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> idxOfMol;
    if(cin.fail()){
      badInput = true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
      continue;
    }

    tmp = World::getInstance().getMolecule(MoleculeById(idxOfMol));
    if(!tmp && idxOfMol!=-1){
      Log() << Verbose(0) << "Invalid Molecule Index" << endl;
      badInput = true;
    }

  } while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return (idxOfMol!=-1);
}


TextDialog::MoleculesTextQuery::MoleculesTextQuery(string title, std::string _description) :
    Dialog::MoleculesQuery(title,_description)
{}

TextDialog::MoleculesTextQuery::~MoleculesTextQuery() {}

bool TextDialog::MoleculesTextQuery::handle() {
  int idxOfMol=-1;
  Log() << Verbose(0) << getTitle();
  std::string line;
  getline(cin,line);
  // dissect by " "
  std::string::iterator olditer = line.begin();
  for(string::iterator iter = line.begin(); iter != line.end(); ++iter) {
    if (*iter == ' ') {
      std::istringstream stream(string(iter, olditer));
      stream >> idxOfMol;
      temp = World::getInstance().getMolecule(MoleculeById(idxOfMol));
      if(!temp && idxOfMol!=-1){
        Log() << Verbose(0) << "Invalid Molecule Index" << idxOfMol << endl;
        break;
      }
      tmp.push_back(temp);
      olditer = iter;
    }
  }
  if (olditer != line.begin()) { // insert last part also
    std::istringstream stream(string(olditer, line.end()));
    stream >> idxOfMol;
    temp = World::getInstance().getMolecule(MoleculeById(idxOfMol));
    if(!temp && idxOfMol!=-1){
      Log() << Verbose(0) << "Invalid Molecule Index" << idxOfMol << endl;
      tmp.push_back(temp);
    }
  }

  return (idxOfMol!=-1);
}

TextDialog::VectorTextQuery::VectorTextQuery(std::string title, bool _check, std::string _description) :
    Dialog::VectorQuery(title,_check,_description)
{}

TextDialog::VectorTextQuery::~VectorTextQuery()
{}

bool TextDialog::VectorTextQuery::handle() {
  std::cout << getTitle();
  const Matrix &M = World::getInstance().getDomain().getM();
  char coords[3] = {'x', 'y', 'z'};
  for (int i=0;i<3;i++)
    std::cout << coords[i] << "[0.." << M.at(i,i) << ( (i!=2) ? "], " : "]: ");

  std::string line;
  getline(cin,line);

  // dissect by ","
  double coord = 0.;
  int counter = 0;
  std::string::iterator olditer = line.begin();
  for(string::iterator iter = line.begin(); (iter != line.end()) && (counter != 3); ++iter) {
    if (*iter == ',') {
      std::istringstream stream(string(iter, olditer));
      stream >> coord;
      tmp[counter++] = coord;
      olditer = iter;
    }
  }
  if ((olditer != line.begin()) && (counter != 3)) { // insert last part also
    std::istringstream stream(string(olditer, line.end()));
    stream >> coord;
    tmp[counter++] = coord;
  }

  // check vector
  return World::getInstance().getDomain().isInside(tmp);
}

TextDialog::VectorsTextQuery::VectorsTextQuery(std::string title, bool _check, std::string _description) :
    Dialog::VectorsQuery(title,_check,_description)
{}

TextDialog::VectorsTextQuery::~VectorsTextQuery()
{}

bool TextDialog::VectorsTextQuery::handle() {
  std::cout << getTitle();
  char coords[3] = {'x', 'y', 'z'};
  const Matrix &M = World::getInstance().getDomain().getM();
  for (int i=0;i<3;i++)
    std::cout << coords[i] << "[0.." << M.at(i,i) << ( (i!=2) ? "], " : "]: ");

  std::string line;
  getline(cin,line);

  // dissect by ","
  double coord = 0.;
  std::string::iterator olditerspace = line.begin();
  std::string::iterator olditercomma = line.begin();
  int counter = 0;
  for(string::iterator vectoriter = line.begin(); vectoriter != line.end(); ++vectoriter) {
    if (*vectoriter == ',')
      counter++;
    if ((*vectoriter == ' ') && (counter == 2)) {
      counter = 0;
      for(string::iterator componentiter = olditerspace; (componentiter != vectoriter) && (counter !=3); ++componentiter) {
        if (*componentiter == ',') {
          std::istringstream stream(string(componentiter, olditercomma));
          stream >> coord;
          temp[counter++] = coord;
          olditercomma = componentiter;
        }
      }
      if ((olditercomma != line.begin()) && (counter != 3)) { // insert last part also
        std::istringstream stream(string(olditercomma, vectoriter));
        stream >> coord;
        temp[counter++] = coord;
      }
      if (World::getInstance().getDomain().isInside(temp))
        tmp.push_back(temp);
      olditerspace = vectoriter;
    }
  }

  return true;
}

TextDialog::BoxTextQuery::BoxTextQuery(std::string title, std::string _description) :
    Dialog::BoxQuery(title,_description)
{}

TextDialog::BoxTextQuery::~BoxTextQuery()
{}

bool TextDialog::BoxTextQuery::handle() {
  Log() << Verbose(0) << getTitle();

  double temp[6];
  std::string coords[6] = {"xx","yx","yy", "zx", "zy", "zz"};
  for (int i=0;i<6;i++) {
    Log() << Verbose(0) << coords[i] << ": ";
    cin >> temp[i];
  }
  Matrix M;
  M.set(0,0, temp[0]);
  M.set(0,1, temp[1]);
  M.set(0,2, temp[2]);
  M.set(1,0, temp[1]);
  M.set(1,1, temp[3]);
  M.set(1,2, temp[4]);
  M.set(2,0, temp[2]);
  M.set(2,1, temp[4]);
  M.set(2,2, temp[5]);
  tmp.setM(M);
  return true;
}

TextDialog::ElementTextQuery::ElementTextQuery(std::string title, std::string _description) :
    Dialog::ElementQuery(title,_description)
{}

TextDialog::ElementTextQuery::~ElementTextQuery()
{}

bool TextDialog::ElementTextQuery::handle() {
  bool badInput=false;
  bool aborted = false;
  const element * temp = NULL;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();

    // try to read as Atomic number
    int Z;
    cin >> Z;
    if(!cin.fail()){
      if(Z==-1){
        aborted = true;
      }
      else{
        temp = World::getInstance().getPeriode()->FindElement(Z);
        if(!temp){
          Log() << Verbose(0) << "No element with this atomic number!" << endl;
          badInput = true;
        }
      }
      continue;
    }
    else{
      cin.clear();
    }

    // Try to read as shorthand
    // the last buffer content was not removed, so we read the
    // same thing again, this time as a std::string
    std::string shorthand;
    cin >> shorthand;
    if(!cin.fail()){
      if(shorthand.empty()){
        aborted = true;
      }
      else{
        temp = World::getInstance().getPeriode()->FindElement(shorthand.c_str());
        if(!temp){
          Log() << Verbose(0) << "No element with this shorthand!" << endl;
          badInput = true;
        }
      }
    }
    else{
      Log() << Verbose(0) << "Could not read input. Try Again." << endl;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      badInput = true;
    }

  }while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return !aborted;
}

TextDialog::ElementsTextQuery::ElementsTextQuery(std::string title, std::string _description) :
    Dialog::ElementsQuery(title,_description)
{}

TextDialog::ElementsTextQuery::~ElementsTextQuery()
{}

bool TextDialog::ElementsTextQuery::handle() {
  std::string shorthand;
  int Z=-1;
  Log() << Verbose(0) << getTitle();
  std::string line;
  getline(cin,line);
  // dissect by " "
  std::string::iterator olditer = line.begin();
  for(string::iterator iter = line.begin(); iter != line.end(); ++iter) {
    if (*iter == ' ') {
      std::istringstream stream(string(iter, olditer));
      stream >> shorthand;
      try {
        Z = lexical_cast<int>(shorthand);
        temp = World::getInstance().getPeriode()->FindElement(Z);
      } catch (bad_lexical_cast) {
        temp = World::getInstance().getPeriode()->FindElement(shorthand.c_str());
      };
      if(!temp && Z!=-1){
        Log() << Verbose(0) << "Invalid Element" << shorthand << endl;
        break;
      }
      tmp.push_back(temp);
      olditer = iter;
    }
  }
  if (olditer != line.begin()) { // insert last part also
    std::istringstream stream(string(olditer, line.end()));
    stream >> shorthand;
    try {
      Z = lexical_cast<int>(shorthand);
      temp = World::getInstance().getPeriode()->FindElement(Z);
    } catch (bad_lexical_cast) {
      temp = World::getInstance().getPeriode()->FindElement(shorthand.c_str());
    };
    if(!temp && Z!=-1) {
      Log() << Verbose(0) << "Invalid Element" << shorthand << endl;
      tmp.push_back(temp);
    }
  }

  return (Z!=-1);
}

TextDialog::FileTextQuery::FileTextQuery(string title, std::string _description) :
    Dialog::FileQuery(title,_description)
{}

TextDialog::FileTextQuery::~FileTextQuery() {}

bool TextDialog::FileTextQuery::handle() {
  Log() << Verbose(0) << getTitle();
  std::string tempstring;
  getline(cin,tempstring);
  tmp = tempstring;
  return true;
}

