/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * Dialog.cpp
 *
 *  Created on: Jan 5, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Dialog.hpp"
#include "Actions/ValueStorage.hpp"

#include "Helpers/Verbose.hpp"
#include "atom.hpp"
#include "Box.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "LinearAlgebra/BoxVector.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "LinearAlgebra/Matrix.hpp"

using namespace std;

Dialog::Dialog()
{
}

Dialog::~Dialog()
{
  list<Query*>::iterator iter;
  for(iter=queries.begin();iter!=queries.end();iter++){
    delete (*iter);
  }
}

void Dialog::registerQuery(Query *query){
  queries.push_back(query);
}

bool Dialog::display(){
  if(checkAll()){
    setAll();
    return true;
  }
  else{
    return false;
  }
}

bool Dialog::checkAll(){
  list<Query*>::iterator iter;
  bool retval = true;
  for(iter=queries.begin(); iter!=queries.end(); iter++){
    retval &= (*iter)->handle();
    // if any query fails (is canceled), we can end the handling process
    if(!retval) {
      DoeLog(1) && (eLog() << Verbose(1) << "The following query failed: " << (**iter).getTitle() << "." << endl);
      break;
    }
  }
  return retval;
}

void Dialog::setAll(){
  list<Query*>::iterator iter;
  for(iter=queries.begin(); iter!=queries.end(); iter++) {
    (*iter)->setResult();
  }
}

bool Dialog::hasQueries(){
  return queries.size();
}

template <> void Dialog::query<void *>(const char *token, std::string description)
{
  queryEmpty(token, description);
}

template <> void Dialog::query<bool>(const char *token, std::string description)
{
  queryBoolean(token, description);
}

template <> void Dialog::query<int>(const char *token, std::string description)
{
  queryInt(token, description);
}

template <> void Dialog::query< std::vector<int> >(const char *token, std::string description)
{
  queryInts(token, description);
}

template <> void Dialog::query<double>(const char *token, std::string description)
{
  queryDouble(token, description);
}

template <> void Dialog::query< std::vector<double> >(const char *token, std::string description)
{
  queryDoubles(token, description);
}

template <> void Dialog::query<std::string>(const char *token, std::string description)
{
  queryString(token, description);
}

template <> void Dialog::query< std::vector<std::string> >(const char *token, std::string description)
{
  queryStrings(token, description);
}

template <> void Dialog::query<atom *>(const char *token, std::string description)
{
  queryAtom(token, description);
}

template <> void Dialog::query< std::vector<atom *> >(const char *token, std::string description)
{
  queryAtoms(token, description);
}

template <> void Dialog::query<molecule *>(const char *token, std::string description)
{
  queryMolecule(token, description);
}

template <> void Dialog::query< std::vector<molecule *> >(const char *token, std::string description)
{
  queryMolecules(token, description);
}

template <> void Dialog::query<Vector>(const char *token, std::string description)
{
  queryVector(token, false, description);
}

template <> void Dialog::query< std::vector<Vector> >(const char *token, std::string description)
{
  queryVectors(token, false, description);
}

template <> void Dialog::query<BoxVector>(const char *token, std::string description)
{
  queryVector(token, true, description);
}

template <> void Dialog::query< std::vector<BoxVector> >(const char *token, std::string description)
{
  queryVectors(token, true, description);
}

template <> void Dialog::query<Box>(const char *token, std::string description)
{
  queryBox(token, description);
}

template <> void Dialog::query<const element *>(const char *token, std::string description)
{
  queryElement(token, description);
}

template <> void Dialog::query< std::vector<const element *> >(const char *token, std::string description)
{
  queryElements(token, description);
}

template <> void Dialog::query< boost::filesystem::path >(const char *token, std::string description)
{
  queryFile(token, description);
}

/****************** Query types Infrastructure **************************/

// Base class
Dialog::Query::Query(string _title, string _description) :
    title(_title),
    description(_description)
{}

Dialog::Query::~Query() {}

const std::string Dialog::Query::getTitle() const{
  return title;
}

const std::string Dialog::Query::getDescription() const{
  return description;
}
// empty Queries

Dialog::EmptyQuery::EmptyQuery(string title, std::string description) :
    Query(title, description)
{}

Dialog::EmptyQuery::~EmptyQuery() {}

void Dialog::EmptyQuery::setResult() {
}

// Int Queries

Dialog::IntQuery::IntQuery(string title, std::string description) :
    Query(title, description)
{}

Dialog::IntQuery::~IntQuery() {}

void Dialog::IntQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Ints Queries

Dialog::IntsQuery::IntsQuery(string title, std::string description) :
    Query(title, description)
{}

Dialog::IntsQuery::~IntsQuery() {}

void Dialog::IntsQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Bool Queries

Dialog::BooleanQuery::BooleanQuery(string title,std::string description) :
    Query(title, description)
{}

Dialog::BooleanQuery::~BooleanQuery() {}

void Dialog::BooleanQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// String Queries

Dialog::StringQuery::StringQuery(string title,std::string _description) :
    Query(title, _description)
{}

Dialog::StringQuery::~StringQuery() {};

void Dialog::StringQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Strings Queries

Dialog::StringsQuery::StringsQuery(string title,std::string _description) :
    Query(title, _description)
{}

Dialog::StringsQuery::~StringsQuery() {};

void Dialog::StringsQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Double Queries

Dialog::DoubleQuery::DoubleQuery(string title, std::string _description) :
    Query(title, _description)
{}

Dialog::DoubleQuery::~DoubleQuery() {};

void Dialog::DoubleQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Doubles Queries

Dialog::DoublesQuery::DoublesQuery(string title, std::string _description) :
    Query(title, _description)
{}

Dialog::DoublesQuery::~DoublesQuery() {};

void Dialog::DoublesQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}


// Atom Queries

Dialog::AtomQuery::AtomQuery(string title, std::string _description) :
    Query(title, _description),
    tmp(0)
{}

Dialog::AtomQuery::~AtomQuery() {}

void Dialog::AtomQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Atoms Queries

Dialog::AtomsQuery::AtomsQuery(string title, std::string _description) :
    Query(title, _description),
    tmp(0)
{}

Dialog::AtomsQuery::~AtomsQuery() {}

void Dialog::AtomsQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Molecule Queries

Dialog::MoleculeQuery::MoleculeQuery(string title, std::string _description) :
    Query(title, _description),
    tmp(0)
{}

Dialog::MoleculeQuery::~MoleculeQuery() {}

void Dialog::MoleculeQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Molecules Queries

Dialog::MoleculesQuery::MoleculesQuery(string title, std::string _description) :
    Query(title, _description),
    tmp(0)
{}

Dialog::MoleculesQuery::~MoleculesQuery() {}

void Dialog::MoleculesQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Vector Queries

Dialog::VectorQuery::VectorQuery(std::string title,bool _check, std::string _description) :
  Query(title, _description),
  check(_check)
{}

Dialog::VectorQuery::~VectorQuery()
{}

void Dialog::VectorQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Vectors Queries

Dialog::VectorsQuery::VectorsQuery(std::string title,bool _check, std::string _description) :
  Query(title, _description),
  check(_check)
{}

Dialog::VectorsQuery::~VectorsQuery()
{}

void Dialog::VectorsQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Box Queries

Dialog::BoxQuery::BoxQuery(std::string title, std::string _description) :
  Query(title, _description)
{}

Dialog::BoxQuery::~BoxQuery()
{}

void Dialog::BoxQuery::setResult() {
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Element Queries
Dialog::ElementQuery::ElementQuery(std::string title, std::string _description) :
  Query(title, _description)
  {}

Dialog::ElementQuery::~ElementQuery(){}

void Dialog::ElementQuery::setResult(){
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// Elements Queries
Dialog::ElementsQuery::ElementsQuery(std::string title, std::string _description) :
  Query(title, _description)
  {}

Dialog::ElementsQuery::~ElementsQuery(){}

void Dialog::ElementsQuery::setResult(){
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}

// File Queries
Dialog::FileQuery::FileQuery(std::string title, std::string _description) :
  Query(title, _description)
  {}

Dialog::FileQuery::~FileQuery(){}

void Dialog::FileQuery::setResult(){
  ValueStorage::getInstance().setCurrentValue(title.c_str(), tmp);
}
