/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/* helper_c_capture.cpp
 *
 * This is inspired by the answer on stackoverflow
 * https://stackoverflow.com/questions/35745541/how-to-get-printed-output-from-ctypes-c-functions-into-jupyter-ipython-notebook
 * where certain libc functions
 * are exposed using a python module (with ctypes). This is however very hacky
 * and as I am using a C++ module anyway, I rather expose these functions cleanly
 * through an extra library.
 *
 * Note:
 *   It was not straight-forward to expose _IO_FILE to boost::python. Hence,
 *   I simply crippled the exposed function to have not arguments and work on
 *   stdout directly.
 *
 * See \b create_c_contextmanager.py
 *
 *  Created on: Apr 15, 2019
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include <boost/python.hpp>

#include <cstdio>
#include <ext/stdio_filebuf.h>
#include <iostream>

#include "CodePatterns/Log.hpp"

using namespace boost::python;

void fflush_stdout() { fflush(stdout); }
void disable_stdout_buffering() { setbuf(stdout, NULL); }
void fflush_stderr() { fflush(stderr); }
void disable_stderr_buffering() { setbuf(stderr, NULL); }

void log_to_fileno(int fileno) {
  __gnu_cxx::stdio_filebuf<char> filebuf(fileno, std::ios::in);
  std::ostream *out = new std::ostream(&filebuf);
  logger::setOutputStream(out);
  errorLogger::setOutputStream(out);
}

BOOST_PYTHON_MODULE(pyMoleCuilder_capture)
{
	def("fflush_stderr", fflush_stderr, "flushing stderr buffer");
	def("fflush_stdout", fflush_stdout, "flushing stdout buffer");
	def("disable_stderr_buffering", disable_stderr_buffering,
			"disabling the error buffer");
	def("disable_stdout_buffering", disable_stdout_buffering,
			"disabling the output buffer");
	def("log_to_fileno", log_to_fileno, "divert log to a given file handle number");
}
