/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the COPYING file or "Copyright notice" in builder.cpp for details.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * TrainingData.cpp
 *
 *  Created on: 15.10.2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "TrainingData.hpp"

#include <iostream>

#include "CodePatterns/toString.hpp"

#include "Fragmentation/SetValues/Fragment.hpp"
#include "FunctionApproximation/FunctionModel.hpp"

void TrainingData::operator()(const range_t &range) {
  for (HomologyContainer::const_iterator iter = range.first; iter != range.second; ++iter) {
    // get distance out of Fragment
    const Fragment &fragment = iter->second.first;
    FunctionModel::arguments_t args = extractor(
          fragment,
          DistanceVector.size()
        );
    DistanceVector.push_back( args );
    const double &energy = iter->second.second;
    EnergyVector.push_back( FunctionModel::results_t(1, energy) );
  }
}

const double TrainingData::getL2Error(const FunctionModel &model) const
{
  double L2sum = 0.;

  FunctionApproximation::inputs_t::const_iterator initer = DistanceVector.begin();
  FunctionApproximation::outputs_t::const_iterator outiter = EnergyVector.begin();
  for (; initer != DistanceVector.end(); ++initer, ++outiter) {
    const FunctionModel::results_t result = model((*initer));
    const double temp = fabs((*outiter)[0] - result[0]);
    L2sum += temp*temp;
  }
  return L2sum;
}

const double TrainingData::getLMaxError(const FunctionModel &model) const
{
  double Lmax = 0.;
  size_t maxindex = -1;
  FunctionApproximation::inputs_t::const_iterator initer = DistanceVector.begin();
  FunctionApproximation::outputs_t::const_iterator outiter = EnergyVector.begin();
  for (; initer != DistanceVector.end(); ++initer, ++outiter) {
    const FunctionModel::results_t result = model((*initer));
    const double temp = fabs((*outiter)[0] - result[0]);
    if (temp > Lmax) {
      Lmax = temp;
      maxindex = std::distance(
          const_cast<const FunctionApproximation::inputs_t &>(DistanceVector).begin(),
          initer
          );
    }
  }
  return Lmax;
}

std::ostream &operator<<(std::ostream &out, const TrainingData &data)
{
  const TrainingData::InputVector_t &DistanceVector = data.getTrainingInputs();
  const TrainingData::OutputVector_t &EnergyVector = data.getTrainingOutputs();
  out << "(" << DistanceVector.size()
      << "," << EnergyVector.size() << ") data pairs: " << std::endl;
  FunctionApproximation::inputs_t::const_iterator initer = DistanceVector.begin();
  FunctionApproximation::outputs_t::const_iterator outiter = EnergyVector.begin();
  for (; initer != DistanceVector.end(); ++initer, ++outiter) {
    for (size_t index = 0; index < (*initer).size(); ++index)
       out << "(" << (*initer)[index].indices.first << "," << (*initer)[index].indices.second
          << ") " << (*initer)[index].distance;
    out << " with energy ";
    out << (*outiter);
    out << std::endl;
  }
  return out;
}
