/*
 * ActionSequenzTest.cpp
 *
 *  Created on: Dec 17, 2009
 *      Author: crueger
 */

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "unittests/ActionSequenceTest.hpp"
#include "Actions/Action.hpp"
#include "Actions/ActionSequence.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ActionSequenceTest );

void setUp();
void tearDown();

void canUndoTest();

/* some neccessary stubs for tests */
class canUndoActionStub : public Action
{
public:
  canUndoActionStub(): Action("canUndoActionStub",false){}
  virtual ~canUndoActionStub(){}

  virtual ActionState* performCall(){
    return Action::success;
  }
  virtual ActionState* performUndo(ActionState*){
    return Action::success;
  }
  virtual ActionState* performRedo(ActionState*){
    return Action::success;
  }
  virtual bool canUndo(){
    return true;
  }
  virtual bool shouldUndo(){
    return true;
  }
};

class cannotUndoActionStub : public Action
{
public:
  cannotUndoActionStub() : Action("cannotUndoActionStub",false){}
  virtual ~cannotUndoActionStub(){}

  virtual ActionState* performCall(){
    return Action::success;
  }
  virtual ActionState* performUndo(ActionState*){
    return Action::success;
  }
  virtual ActionState* performRedo(ActionState*){
    return Action::success;
  }
  virtual bool canUndo(){
    return false;
  }
  virtual bool shouldUndo(){
   return true;
  }
};

class wasCalledActionStub : public Action
{
public:
  wasCalledActionStub() :
      Action("wasCalledActionStub",false),
      called(false)
  {}
  virtual ~wasCalledActionStub(){}

  virtual ActionState* performCall(){
    called = true;
    return Action::success;
  }
  virtual ActionState* performUndo(ActionState*){
    called = false;
    return Action::success;
  }
  virtual ActionState* performRedo(ActionState*){
    called = true;
    return Action::success;
  }
  virtual bool canUndo(){
    return true;
  }
  virtual bool shouldUndo(){
    return true;
  }
  bool wasCalled(){
    return called;
  }
private:
  bool called;
};

void ActionSequenceTest::setUp(){
  // create some necessary stubs used in this test
  positive1 = new canUndoActionStub();
  positive2 = new canUndoActionStub();
  negative1 = new cannotUndoActionStub();
  negative2 = new cannotUndoActionStub();

  shouldCall1 = new wasCalledActionStub();
  shouldCall2 = new wasCalledActionStub();
  shouldNotCall1 = new wasCalledActionStub();
  shouldNotCall2 = new wasCalledActionStub();

}

void ActionSequenceTest::tearDown(){
  delete positive1;
  delete positive2;
  delete negative1;
  delete negative2;

  delete shouldCall1;
  delete shouldCall2;
  delete shouldNotCall1;
  delete shouldNotCall2;

}

void ActionSequenceTest::canUndoTest(){
  // first section:
  {
    // test some combinations
    {
      ActionSequence *sequence = new ActionSequence();
      sequence->addAction(positive1);
      sequence->addAction(positive2);
      CPPUNIT_ASSERT_EQUAL( true, sequence->canUndo() );
      delete sequence;
    }
    {
      ActionSequence *sequence = new ActionSequence();
      sequence->addAction(positive1);
      sequence->addAction(negative2);
      CPPUNIT_ASSERT_EQUAL( false, sequence->canUndo() );
      delete sequence;
    }
    {
      ActionSequence *sequence = new ActionSequence();
      sequence->addAction(negative1);
      sequence->addAction(positive2);
      CPPUNIT_ASSERT_EQUAL( false, sequence->canUndo() );
      delete sequence;
    }
    {
      ActionSequence *sequence = new ActionSequence();
      sequence->addAction(negative1);
      sequence->addAction(negative2);
      CPPUNIT_ASSERT_EQUAL( false, sequence->canUndo() );
      delete sequence;
    }
  }

  // second section:
  {
    // empty sequence can be undone
    ActionSequence *sequence = new ActionSequence();
    CPPUNIT_ASSERT_EQUAL( true, sequence->canUndo() );
    // if only a positive action is contained it can be undone
    sequence->addAction(positive1);
    CPPUNIT_ASSERT_EQUAL( true, sequence->canUndo() );
    // the single negative action should block the process
    sequence->addAction(negative1);
    CPPUNIT_ASSERT_EQUAL( false, sequence->canUndo() );
    // after removing the negative action all is well again
    sequence->removeLastAction();
    CPPUNIT_ASSERT_EQUAL( true, sequence->canUndo() );
    delete sequence;
  }
}

void ActionSequenceTest::doesCallTest(){
  ActionSequence *sequence = new ActionSequence();
  sequence->addAction(shouldCall1);
  sequence->addAction(shouldCall2);
  sequence->addAction(shouldNotCall1);
  sequence->addAction(shouldNotCall2);
  sequence->removeLastAction();
  sequence->removeLastAction();

  sequence->callAll();

  CPPUNIT_ASSERT_EQUAL(true,shouldCall1->wasCalled());
  CPPUNIT_ASSERT_EQUAL(true,shouldCall2->wasCalled());
  CPPUNIT_ASSERT_EQUAL(false,shouldNotCall1->wasCalled());
  CPPUNIT_ASSERT_EQUAL(false,shouldNotCall2->wasCalled());

  delete sequence;
}

void ActionSequenceTest::doesUndoTest(){
  ActionSequence *sequence = new ActionSequence();
  sequence->addAction(shouldNotCall1);
  sequence->addAction(shouldNotCall2);
  sequence->addAction(shouldCall1);
  sequence->addAction(shouldCall2);

  ActionSequence::stateSet states = sequence->callAll();

  sequence->removeLastAction();
  sequence->removeLastAction();

  sequence->undoAll(states);

  CPPUNIT_ASSERT_EQUAL(true,shouldCall1->wasCalled());
  CPPUNIT_ASSERT_EQUAL(true,shouldCall2->wasCalled());
  CPPUNIT_ASSERT_EQUAL(false,shouldNotCall1->wasCalled());
  CPPUNIT_ASSERT_EQUAL(false,shouldNotCall2->wasCalled());

  delete sequence;
}


