#!/bin/bash

optimizations=("-g3"\
               "-O0"\
	       "-O1"\
               "-O2"\
               "-O3"\
               "-Os");

options=("" \
         "-DLOG_OBSERVER" \
         "-DNO_MEMDEBUG" \
         "-DNO_CACHING" \
         "-DNDEBUG" \
         "-DNO_MEMDEBUG -DLOG_OBSERVER" \
         "-DNO_CACHING -DLOG_OBSERVER" \
         "-DNO_CACHING -DNO_MEMDEBUG" \
         "-DNDEBUG -DNO_CACHING" \
	 "-DNO_CACHING -DNO_MEMDEBUG -DLOG_OBSERVER" \
        );

outfile="test.log";
logfile="full.log";
noprocs=1;
docheck=0;
docheck_mem=0;
if [ -n "$TMPDIR" ]
then
  tmpdir="$TMPDIR";
else
  tmpdir="/tmp";
fi
tmppattern="MolecuilderTest";

function usage(){
  echo "usage $0 options";
  echo "";
  echo "This script runs a full test for molecuilder, using several compilation options";
  echo "";
  echo "OPTIONS:";
  echo "  -h			Show this message"
  echo "  -o <outfile>		Outfile to use for test results";
  echo "  -f <logfile>		File to use for output from commands";
  echo "  -j <no_proc>          Parallel compiling and tests";
  echo "  -s			Short tests (no memcheck)";
  echo "  -c            	Only configure and compile (implies -s)";
  echo "  -O <opt-level>	Only compile this optimization level";
  echo "  -t <tmpDir>          	Use tmpDir as temporary directory";
  echo "  -p <prefix>		Prefix to use for directory names (standard MolecuilderTest)";
}

function logdate()
{
  echo -e "Launched on `date`.\n\n" >> $logfile 2>&1;
}


while getopts “ho:f:j:scO:t:p:” OPTION
do
  case $OPTION in
   h)
      usage;
      exit 0;
      ;;
   o)
     outfile="$OPTARG";
     ;;
   f)
     logfile="$OPTARG";
     ;;
   j)
     noprocs=("$OPTARG");
     ;;
   s)
     docheck_mem=1;
     ;;
   c)
     docheck=1;
     docheck_mem=1;
     ;;
   O)
     optimizations=("-O$OPTARG");
     ;;
   t)
     tmpdir="$OPTARG";
     ;;
   p)
     tmppattern="$OPTARG";
     ;;
   ?)
     usage;
     exit 1;
     ;;
  esac
done

# test if all arguments were provided
if [[ -z "$outfile" ]] || [[ -z "$logfile" ]] || [[ -z "$tmpdir" ]] || [[ -z "$tmppattern" ]]
then
  usage;
  exit 1;
fi

# turn all relative paths into absolutes
outfile=`realpath -s $outfile`;
logfile=`realpath -s $logfile`;
tmpdir=`realpath -s $tmpdir`;


BOOST_ROOT=/opt/packages/boost;

function configure(){
  echo "Configuring";
  $1/configure $3 --prefix=$PWD CXXFLAGS="-Wall $2" >> $logfile 2>&1;
}

function compile(){
  echo "Making";
  if [ $noprocs -gt 1 ]; then
    make -j$noprocs all >>$logfile 2>&1;
  else
    make all >>$logfile 2>&1;
  fi
}

function check(){
  echo "Checking";
  if [ $noprocs -gt 1 ]; then
    make -j$noprocs check >> $logfile 2>&1;
  else
    make check >> $logfile 2>&1;
  fi
}

function memcheck(){
  echo "Valgrinding";
  retval=0;
  for test in src/unittests/*
  do
    if [ -x "$test" ]
    then
      echo -n "    $test: " >> $outfile;
      valgrind -v --leak-check=full --leak-resolution=high --show-reachable=yes --error-exitcode=255 $test >> $logfile 2>&1;
      if [ $? ]
      then
      	echo "OK" >> $outfile
      else
        echo "FAIL" >> $outfile;
        retval=1;
      fi
    fi
  done
  return $retval
}

function test(){

  echo "Testing with \"$2 $3\"";
  echo "" >> $logfile;
  echo "" >> $logfile;
  echo "" >> $logfile;
  echo "Testing with \"$2 $3\"" >> $logfile;

  if [ x"$2" == x"-g3" ]
  then
    valgrind="--with-valgrind"
  else
    valgrind="--without-valgrind"
  fi
  echo -n "  Configuring $valgrind: " >> $outfile;
  if configure "$1" "$2 $3" "$valgrind"
  then
    echo "OK" >> $outfile;
  else
    echo "FAIL" >> $outfile;
    return;
  fi

  echo -n "  Compiling: " >> $outfile;
  if compile
  then
    echo "OK" >> $outfile;
  else
    echo "FAIL" >> $outfile;
    return;
  fi

  if [ $docheck ]
  then
    echo -n "  Running testsuite: " >> $outfile;
    if check
    then
      echo "OK" >> $outfile;
    else
      echo "FAIL" >> $outfile;
      return;
    fi
  fi

  if [ $docheck_mem ]
  then
    echo  "  Checking memory Errors:..." >> $outfile;
    if memcheck
    then
      echo "  ...OK" >> $outfile
    else
      echo "  ...FAIL" >> $outfile
      return
    fi
  fi
}



function run(){
  echo "Testing with \"$1 $2\":" >> $outfile;
  testdir=`mktemp -d --tmpdir=$tmpdir $tmppattern.XXXXXXXXXX`;
  basedir=$PWD;
  cd $testdir;
  test "$basedir" "$1" "$2";
  cd $basedir;
  rm -rf $testdir;
  echo "" >> $outfile;
}


echo -n "Full compilation test for Molecuilder started on " > $outfile;
date >> $outfile;
echo "" > $logfile;

for optimization in "${optimizations[@]}"
do
  for option in "${options[@]}"
  do
    run "$optimization" "$option";
  done
done

echo -n "Full compilation test for Molecuilder on " >> $outfile
date >> $outfile
