/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * listofbondsunittest.cpp
 *
 *  Created on: 18 Oct 2009
 *      Author: user
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <cstring>

#include "listofbondsunittest.hpp"

#include "World.hpp"
#include "atom.hpp"
#include "bond.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "World.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ListOfBondsTest );


void ListOfBondsTest::setUp()
{
  atom *Walker = NULL;

  // construct element
  hydrogen = World::getInstance().getPeriode()->FindElement(1);
  CPPUNIT_ASSERT(hydrogen != NULL && "could not find element hydrogen");

  // construct molecule (tetraeder of hydrogens)
  TestMolecule = World::getInstance().createMolecule();
  CPPUNIT_ASSERT(TestMolecule != NULL && "could not create molecule");
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(1., 0., 1. ));
  TestMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(0., 1., 1. ));
  TestMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(1., 1., 0. ));
  TestMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(0., 0., 0. ));
  TestMolecule->AddAtom(Walker);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule->getAtomCount(), 4 );
};


void ListOfBondsTest::tearDown()
{
  // remove
  World::getInstance().destroyMolecule(TestMolecule);
  // note that all the atoms, molecules, the tafel and the elements
  // are all cleaned when the world is destroyed
  World::purgeInstance();
  logger::purgeInstance();
};

/** Tests whether setup worked correctly.
 *
 */
void ListOfBondsTest::SetupTest()
{
  CPPUNIT_ASSERT_EQUAL( false, TestMolecule->empty() );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, TestMolecule->size() );
};

/** Unit Test of molecule::AddBond()
 *
 */
void ListOfBondsTest::AddingBondTest()
{
  bond *Binder = NULL;
  molecule::iterator iter = TestMolecule->begin();
  atom *atom1 = *iter;
  iter++;
  atom *atom2 = *iter;
  CPPUNIT_ASSERT( atom1 != NULL );
  CPPUNIT_ASSERT( atom2 != NULL );

  // add bond
  Binder = TestMolecule->AddBond(atom1, atom2, 1);
  CPPUNIT_ASSERT( Binder != NULL );
  CPPUNIT_ASSERT_EQUAL ( true, TestMolecule->hasBondStructure() );

  // check that bond contains the two atoms
  CPPUNIT_ASSERT_EQUAL( true, Binder->Contains(atom1) );
  CPPUNIT_ASSERT_EQUAL( true, Binder->Contains(atom2) );

  // check that bond is present in both atoms
  bond *TestBond1 = *(atom1->ListOfBonds.begin());
  CPPUNIT_ASSERT_EQUAL( TestBond1, Binder );
  bond *TestBond2 = *(atom2->ListOfBonds.begin());
  CPPUNIT_ASSERT_EQUAL( TestBond2, Binder );
};

/** Unit Test of molecule::RemoveBond()
 *
 */
void ListOfBondsTest::RemovingBondTest()
{
  bond *Binder = NULL;
  molecule::iterator iter = TestMolecule->begin();
  atom *atom1 = *iter;
  iter++;
  atom *atom2 = *iter;
  CPPUNIT_ASSERT( atom1 != NULL );
  CPPUNIT_ASSERT( atom2 != NULL );

  // add bond
  Binder = TestMolecule->AddBond(atom1, atom2, 1);
  CPPUNIT_ASSERT( Binder != NULL );

  // remove bond
  TestMolecule->RemoveBond(Binder);

  // check if removed from atoms
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom1->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom2->ListOfBonds.size() );

  // check if removed from molecule
  CPPUNIT_ASSERT_EQUAL( false, TestMolecule->hasBondStructure() );
};

/** Unit Test of molecule::RemoveBonds()
 *
 */
void ListOfBondsTest::RemovingBondsTest()
{
  bond *Binder = NULL;
  molecule::iterator iter = TestMolecule->begin();
  atom *atom1 = *iter;
  iter++;
  atom *atom2 = *iter;
  iter++;
  atom *atom3 = *iter;
  CPPUNIT_ASSERT( atom1 != NULL );
  CPPUNIT_ASSERT( atom2 != NULL );
  CPPUNIT_ASSERT( atom3 != NULL );

  // add bond
  Binder = TestMolecule->AddBond(atom1, atom2, 1);
  CPPUNIT_ASSERT( Binder != NULL );
  Binder = TestMolecule->AddBond(atom1, atom3, 1);
  CPPUNIT_ASSERT( Binder != NULL );
  Binder = TestMolecule->AddBond(atom2, atom3, 1);
  CPPUNIT_ASSERT( Binder != NULL );

  // check that all are present
  CPPUNIT_ASSERT_EQUAL( (size_t) 2, atom1->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 2, atom2->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 2, atom3->ListOfBonds.size() );

  // remove bond
  TestMolecule->RemoveBonds(atom1);

  // check if removed from atoms
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom1->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 1, atom2->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 1, atom3->ListOfBonds.size() );

  // check if removed from molecule
  CPPUNIT_ASSERT_EQUAL( true, TestMolecule->hasBondStructure() );
  CPPUNIT_ASSERT_EQUAL( (unsigned int)1, TestMolecule->CountBonds() );
};

/** Unit Test of delete(bond *)
 *
 */
void ListOfBondsTest::DeleteBondTest()
{
  bond *Binder = NULL;
  molecule::iterator iter = TestMolecule->begin();
  atom *atom1 = *iter;
  iter++;
  atom *atom2 = *iter;
  CPPUNIT_ASSERT( atom1 != NULL );
  CPPUNIT_ASSERT( atom2 != NULL );

  // add bond
  Binder = TestMolecule->AddBond(atom1, atom2, 1);
  CPPUNIT_ASSERT( Binder != NULL );

  // remove bond
  delete(Binder);

  // check if removed from atoms
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom1->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom2->ListOfBonds.size() );

  // check if removed from molecule
  CPPUNIT_ASSERT_EQUAL( false, TestMolecule->hasBondStructure() );
};

/** Unit Test of molecule::RemoveAtom()
 *
 */
void ListOfBondsTest::RemoveAtomTest()
{
  bond *Binder = NULL;
  molecule::iterator iter = TestMolecule->begin();
  atom *atom1 = *iter;
  iter++;
  atom *atom2 = *iter;
  CPPUNIT_ASSERT( atom1 != NULL );
  CPPUNIT_ASSERT( atom2 != NULL );

  // add bond
  Binder = TestMolecule->AddBond(atom1, atom2, 1);
  CPPUNIT_ASSERT( Binder != NULL );

  // remove atom2
  TestMolecule->RemoveAtom(atom2);

  // check bond if removed from other atom
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom1->ListOfBonds.size() );

  // check if removed from molecule
  CPPUNIT_ASSERT_EQUAL( false, TestMolecule->hasBondStructure() );
};

/** Unit Test of delete(atom *)
 *
 */
void ListOfBondsTest::DeleteAtomTest()
{
  atom *atom1 = NULL;
  atom *atom2 = NULL;
  bond *Binder = NULL;
  {
    molecule::iterator iter = TestMolecule->begin();
    atom1 = *iter;
    iter++;
    atom2 = *iter;
  }
  CPPUNIT_ASSERT( atom1 != NULL );
  CPPUNIT_ASSERT( atom2 != NULL );

  // add bond
  Binder = TestMolecule->AddBond(atom1, atom2, 1);
  CPPUNIT_ASSERT( Binder != NULL );

  CPPUNIT_ASSERT_EQUAL( (size_t) 1, atom1->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 1, atom2->ListOfBonds.size() );

  CPPUNIT_ASSERT_EQUAL( true, TestMolecule->hasBondStructure() );

  // remove atom2
  World::getInstance().destroyAtom(atom2);

  // check bond if removed from other atom
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom1->ListOfBonds.size() );

  // check if removed from molecule
  CPPUNIT_ASSERT_EQUAL( false, TestMolecule->hasBondStructure() );
};
