/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * gslvectorunittest.cpp
 *
 *  Created on: Jan 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "gslvectorunittest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( GSLVectorTest );


void GSLVectorTest::setUp()
{
  v = new GSLVector(3);
};

void GSLVectorTest::tearDown()
{
  delete(v);
};

/** Unit test for accessing elements in the vector.
 *
 */
void GSLVectorTest::AccessTest()
{
  // set
  for (int i=0;i<3;i++)
    v->Set(i,i);

  // and check
  double *ptr = NULL;
  for (int i=0;i<3;i++) {
    CPPUNIT_ASSERT_EQUAL( (double)i, v->Get(i) );
    ptr = v->Pointer(i);
    CPPUNIT_ASSERT_EQUAL( (double)i, *ptr );
  }

  // out of bounds
  //CPPUNIT_ASSERT_EQUAL(0., v->Get(3) );
  //CPPUNIT_ASSERT_EQUAL(0., v->Get(-1) );
};


/** Unit test for initializing the vector.
 *
 */
void GSLVectorTest::InitializaionTest()
{
  // set zero
  v->SetZero();
  for (int i=0;i<3;i++)
    CPPUNIT_ASSERT_EQUAL( 0., v->Get(i) );

  // set basis
  for (int i=0;i<3;i++) {
    v->SetBasis(i);
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( i == j ? 1. : 0. , v->Get(j) );
  }

  // set all
  v->SetAll(1.5);
  for (int i=0;i<3;i++)
    CPPUNIT_ASSERT_EQUAL( 1.5, v->Get(i) );
};

/** Unit test for copying vectors.
 *
 */
void GSLVectorTest::CopyTest()
{
  // set basis
  GSLVector *dest = NULL;
  for (int i=0;i<3;i++) {
    v->SetBasis(i);
    dest = new GSLVector(v);
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( v->Get(j) , dest->Get(j) );

    delete(dest);
  }
};

/** Unit test for exchanging elements of a vector.
 *
 */
void GSLVectorTest::ExchangeTest()
{
  // set basis
  for (int i=0;i<3;i++) {
    v->SetBasis(i);
    v->SwapElements((i)%3,(i+1)%3);
    for (int j=0;j<3;j++)
      CPPUNIT_ASSERT_EQUAL( (i+1)%3 == j ? 1. : 0. , v->Get(j) );
  }

  // set to 1,2,3 and reverse
  for (int i=0;i<3;i++)
    v->Set(i,i+1);
  v->Reverse();
  for (int j=0;j<3;j++)
    CPPUNIT_ASSERT_EQUAL( (double)(3-j), v->Get(j) );
};

/** UnitTest for operators.
 */
void GSLVectorTest::OperatorIsTest()
{
  GSLVector zero(3);
  GSLVector unit(3);
  zero.SetZero();
  unit.SetZero();
  unit.Set(1,1.);
  // summation and scaling
  CPPUNIT_ASSERT_EQUAL( true, unit.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, zero.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, unit.IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, zero.IsZero() );
};

/** UnitTest for operators.
 */
void GSLVectorTest::OperatorAlgebraTest()
{
  GSLVector zero(3);
  GSLVector unit(3);
  zero.SetZero();
  unit.SetZero();
  unit.Set(1,1.);
  // summation and scaling
  CPPUNIT_ASSERT_EQUAL( true, (zero+unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (zero+unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (zero-unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (zero-unit).IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, (zero+zero).IsZero() );
  CPPUNIT_ASSERT_EQUAL( false, (unit*0.98).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (0.98*unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (unit*1.).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (1.*unit).IsOne() );

  CPPUNIT_ASSERT_EQUAL( unit, (zero+unit) );
  CPPUNIT_ASSERT_EQUAL( zero, (zero+zero) );
  CPPUNIT_ASSERT_EQUAL( unit, (unit+zero) );

  unit += zero;
  CPPUNIT_ASSERT_EQUAL( true, unit.IsOne() );
  unit *= 1.;
  CPPUNIT_ASSERT_EQUAL( true, unit.IsOne() );
};
