/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * bondgraphunittest.cpp
 *
 *  Created on: Oct 29, 2009
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <iostream>
#include <stdio.h>
#include <cstring>

#include "Helpers/Assert.hpp"

#include "World.hpp"
#include "atom.hpp"
#include "bond.hpp"
#include "bondgraph.hpp"
#include "element.hpp"
#include "Helpers/Log.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "bondgraphunittest.hpp"
#include "World.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( BondGraphTest );


void BondGraphTest::setUp()
{
  atom *Walker = NULL;

  // construct element
  hydrogen = World::getInstance().getPeriode()->FindElement(1);
  carbon = World::getInstance().getPeriode()->FindElement(6);
  CPPUNIT_ASSERT(hydrogen != NULL && "could not find element hydrogen");
  CPPUNIT_ASSERT(carbon != NULL && "could not find element carbon");

  // construct molecule (tetraeder of hydrogens)
  TestMolecule = World::getInstance().createMolecule();
  CPPUNIT_ASSERT(TestMolecule != NULL && "could not create molecule");
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(carbon);
  Walker->setPosition(Vector(1., 0., 1. ));
  TestMolecule->AddAtom(Walker);

  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(carbon);
  Walker->setPosition(Vector(0., 1., 1. ));
  TestMolecule->AddAtom(Walker);

  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(carbon);
  Walker->setPosition(Vector(1., 1., 0. ));
  TestMolecule->AddAtom(Walker);

  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(carbon);
  Walker->setPosition(Vector(0., 0., 0. ));
  TestMolecule->AddAtom(Walker);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule->getAtomCount(), 4 );

  // create a small file with table
  dummyname = new string("dummy.dat");
  CPPUNIT_ASSERT(dummyname != NULL && "could not create string");
  filename = new string("test.dat");
  CPPUNIT_ASSERT(filename != NULL && "could not create string");
  ofstream test(filename->c_str());
  test << ".\tH\tHe\tLi\tBe\tB\tC\n";
  test << "H\t1.\t1.\t1.\t1.\t1.\t1.2\n";
  test << "He\t1.\t1.\t1.\t1.\t1.\t1.\n";
  test << "Li\t1.\t1.\t1.\t1.\t1.\t1.\n";
  test << "Be\t1.\t1.\t1.\t1.\t1.\t1.\n";
  test << "B\t1.\t1.\t1.\t1.\t1.\t1.\n";
  test << "C\t1.2\t1.\t1.\t1.\t1.\t1.5\n";
  test.close();
  BG = new BondGraph(true);
  CPPUNIT_ASSERT(BG != NULL && "could not create BondGraph");
};


void BondGraphTest::tearDown()
{
  // remove the file
  remove(filename->c_str());
  delete(filename);
  delete(dummyname);
  delete(BG);

  // remove molecule
  World::getInstance().destroyMolecule(TestMolecule);
  // note that all the atoms, molecules, the tafel and the elements
  // are all cleaned when the world is destroyed
  World::purgeInstance();
  logger::purgeInstance();
};

/** Tests whether setup worked.
 */
void BondGraphTest::SetupTest()
{
  CPPUNIT_ASSERT_EQUAL (false, TestMolecule->empty());
  CPPUNIT_ASSERT_EQUAL ((size_t)4, TestMolecule->size());
};

/** UnitTest for BondGraphTest::LoadBondLengthTable().
 */
void BondGraphTest::LoadTableTest()
{
  CPPUNIT_ASSERT_EQUAL( true , BG->LoadBondLengthTable(*filename) );
  CPPUNIT_ASSERT_EQUAL( 1., BG->GetBondLength(0,0) );
  CPPUNIT_ASSERT_EQUAL( 1.2, BG->GetBondLength(0,5) );
  CPPUNIT_ASSERT_EQUAL( 1.5, BG->GetBondLength(5,5) );
};

/** UnitTest for BondGraphTest::ConstructBondGraph().
 */
void BondGraphTest::ConstructGraphFromTableTest()
{
  molecule::iterator Walker = TestMolecule->begin();
  molecule::iterator Runner = TestMolecule->begin();
  Runner++;
  CPPUNIT_ASSERT_EQUAL( true , BG->LoadBondLengthTable(*filename) );
  CPPUNIT_ASSERT_EQUAL( true , BG->ConstructBondGraph(TestMolecule) );
  CPPUNIT_ASSERT_EQUAL( true , (*Walker)->IsBondedTo((*Runner)) );
};

/** UnitTest for BondGraphTest::ConstructBondGraph().
 */
void BondGraphTest::ConstructGraphFromCovalentRadiiTest()
{

  //atom *Walker = TestMolecule->start->next;
  //atom *Runner = TestMolecule->end->previous;
  //CPPUNIT_ASSERT( TestMolecule->end != Walker );
  CPPUNIT_ASSERT_EQUAL( false , BG->LoadBondLengthTable(*dummyname) );
  CPPUNIT_ASSERT_EQUAL( true , BG->ConstructBondGraph(TestMolecule) );

  // this cannot be assured using dynamic IDs
  //CPPUNIT_ASSERT_EQUAL( true , Walker->IsBondedTo(Runner) );
};

