/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * atomsCalculationTest.cpp
 *
 *  Created on: Feb 19, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "atomsCalculationTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>
#include <iostream>
#include <boost/bind.hpp>

#include "Descriptors/AtomDescriptor.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Actions/AtomsCalculation.hpp"
#include "Actions/AtomsCalculation_impl.hpp"
#include "Actions/ActionRegistry.hpp"

#include "World.hpp"
#include "World_calculations.hpp"
#include "atom.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( atomsCalculationTest );

// set up and tear down
void atomsCalculationTest::setUp(){
  World::getInstance();
  for(int i=0;i<ATOM_COUNT;++i){
    atoms[i]= World::getInstance().createAtom();
    atomIds[i]= atoms[i]->getId();
  }
}
void atomsCalculationTest::tearDown(){
  World::purgeInstance();
  ActionRegistry::purgeInstance();
}

// some helper functions
static bool hasAllIds(std::vector<atomId_t> atoms,atomId_t ids[ATOM_COUNT], std::set<atomId_t> excluded = std::set<atomId_t>()){
  for(int i=0;i<ATOM_COUNT;++i){
    atomId_t id = ids[i];
    if(!excluded.count(id)){
      std::vector<atomId_t>::iterator iter;
      bool res=false;
      for(iter=atoms.begin();iter!=atoms.end();++iter){
        res |= (*iter) == id;
      }
      if(!res) {
        cout << "Atom " << id << " missing in returned list" << endl;
        return false;
      }
    }
  }
  return true;
}

static bool hasNoDuplicates(std::vector<atomId_t> ids){
  std::set<atomId_t> found;
  std::vector<atomId_t>::iterator iter;
  for(iter=ids.begin();iter!=ids.end();++iter){
    int id = (*iter);
    if(found.count(id))
      return false;
    found.insert(id);
  }
  return true;
}

void atomsCalculationTest::testCalculateSimple(){
  AtomsCalculation<atomId_t> *calc = World::getInstance().calcOnAtoms<atomId_t>(boost::bind(&atom::getId,_1),"FOO",AllAtoms());
  std::vector<atomId_t> allIds = (*calc)();
  CPPUNIT_ASSERT(hasAllIds(allIds,atomIds));
  CPPUNIT_ASSERT(hasNoDuplicates(allIds));
}

void atomsCalculationTest::testCalculateExcluded(){
  atomId_t excluded = atomIds[ATOM_COUNT/2];
  AtomsCalculation<atomId_t> *calc = World::getInstance().calcOnAtoms<atomId_t>(boost::bind(&atom::getId,_1),"FOO",AllAtoms() && !AtomById(excluded));
  std::vector<atomId_t> allIds = (*calc)();
  std::set<atomId_t> excluded_set;
  excluded_set.insert(excluded);
  CPPUNIT_ASSERT(hasAllIds(allIds,atomIds,excluded_set));
  CPPUNIT_ASSERT(hasNoDuplicates(allIds));
  CPPUNIT_ASSERT_EQUAL((size_t)(ATOM_COUNT-1),allIds.size());
}
