/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * analysisbondsunittest.cpp
 *
 *  Created on: Nov 7, 2009
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <iostream>
#include <stdio.h>
#include <cstring>

#include "analysis_bonds.hpp"
#include "analysisbondsunittest.hpp"
#include "atom.hpp"
#include "bond.hpp"
#include "bondgraph.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "World.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AnalysisBondsTest );


void AnalysisBondsTest::setUp()
{
  atom *Walker = NULL;

  // get elements
  hydrogen = World::getInstance().getPeriode()->FindElement(1);
  carbon = World::getInstance().getPeriode()->FindElement(6);
  CPPUNIT_ASSERT(hydrogen != NULL && "could not find element hydrogen");
  CPPUNIT_ASSERT(carbon != NULL && "could not find element carbon");

  // construct molecule (tetraeder of hydrogens)
  TestMolecule = World::getInstance().createMolecule();
  CPPUNIT_ASSERT(TestMolecule != NULL && "could not create molecule");
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(1.5, 0., 1.5 ));
  TestMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(0., 1.5, 1.5 ));
  TestMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(1.5, 1.5, 0. ));
  TestMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(0., 0., 0. ));
  TestMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
  Walker->setType(carbon);
  Walker->setPosition(Vector(0.5, 0.5, 0.5 ));
  TestMolecule->AddAtom(Walker);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule->getAtomCount(), 5 );

  // create a small file with table
  filename = new string("test.dat");
  CPPUNIT_ASSERT(filename != NULL && "could not create string");
  ofstream test(filename->c_str());
  test << ".\tH\tHe\tLi\tBe\tB\tC\n";
  test << "H\t1.\t1.\t1.\t1.\t1.\t1.2\n";
  test << "He\t1.\t1.\t1.\t1.\t1.\t1.\n";
  test << "Li\t1.\t1.\t1.\t1.\t1.\t1.\n";
  test << "Be\t1.\t1.\t1.\t1.\t1.\t1.\n";
  test << "B\t1.\t1.\t1.\t1.\t1.\t1.\n";
  test << "C\t1.2\t1.\t1.\t1.\t1.\t1.5\n";
  test.close();
  BG = new BondGraph(true);
  CPPUNIT_ASSERT(BG != NULL && "could not create BondGraph");

  CPPUNIT_ASSERT_EQUAL( true , BG->LoadBondLengthTable(*filename) );
  CPPUNIT_ASSERT_EQUAL( 1., BG->GetBondLength(0,0) );
  CPPUNIT_ASSERT_EQUAL( 1.2, BG->GetBondLength(0,5) );
  CPPUNIT_ASSERT_EQUAL( 1.5, BG->GetBondLength(5,5) );

  BG->ConstructBondGraph(TestMolecule);
};


void AnalysisBondsTest::tearDown()
{
  // remove the file
  remove(filename->c_str());
  delete(filename);
  delete(BG);

  // remove molecule
  World::getInstance().destroyMolecule(TestMolecule);
  // note that all the atoms are cleaned by TestMolecule
  World::purgeInstance();
};

/** UnitTest for GetMaxMinMeanBondCount().
 */
void AnalysisBondsTest::GetMaxMinMeanBondCountTest()
{
  double Min = 20.; // check that initialization resets these arbitrary values
  double Mean = 200.;
  double Max = 1e-6;
  GetMaxMinMeanBondCount(TestMolecule, Min, Mean, Max);
  CPPUNIT_ASSERT_EQUAL( 1., Min );
  CPPUNIT_ASSERT_EQUAL( 1.6, Mean );
  CPPUNIT_ASSERT_EQUAL( 4., Max );

};

/** UnitTest for MinMaxBondDistanceBetweenElements().
 */
void AnalysisBondsTest::MinMeanMaxBondDistanceBetweenElementsTest()
{
  double Min = 20.; // check that initialization resets these arbitrary values
  double Mean = 2e+6;
  double Max = 1e-6;
  double Min2 = 20.;
  double Mean2 = 2e+6;
  double Max2 = 1e-6;
  const double maxbondlength = sqrt(1.*1. + 1.*1. + .5*.5);
  const double minbondlength = sqrt(.5*.5 + .5*.5 + .5*.5);
  const double meanbondlength = (minbondlength+3.*maxbondlength)/4.;
  // check bond lengths C-H
  MinMeanMaxBondDistanceBetweenElements(TestMolecule, hydrogen, carbon, Min, Mean, Max);
  CPPUNIT_ASSERT_EQUAL( minbondlength , Min );
  CPPUNIT_ASSERT_EQUAL( meanbondlength , Mean );
  CPPUNIT_ASSERT_EQUAL( maxbondlength , Max );

  // check that elements are symmetric, i.e. C-H == H-C
  MinMeanMaxBondDistanceBetweenElements(TestMolecule, carbon, hydrogen, Min2, Mean2, Max2);
  CPPUNIT_ASSERT_EQUAL( Min , Min2 );
  CPPUNIT_ASSERT_EQUAL( Mean , Mean2 );
  CPPUNIT_ASSERT_EQUAL( Max , Max2 );

  // check no bond case (no bonds H-H in system!)
  MinMeanMaxBondDistanceBetweenElements(TestMolecule, hydrogen, hydrogen, Min, Mean, Max);
  CPPUNIT_ASSERT_EQUAL( 0. , Min );
  CPPUNIT_ASSERT_EQUAL( 0. , Mean );
  CPPUNIT_ASSERT_EQUAL( 0. , Max );
};
