/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * FormulaUnittest.cpp
 *
 *  Created on: Jul 21, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FormulaUnittest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <iostream>

#include "Formula.hpp"
#include "World.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace std;

CPPUNIT_TEST_SUITE_REGISTRATION( FormulaUnittest );

void FormulaUnittest::setUp(){}
void FormulaUnittest::tearDown(){
  World::purgeInstance();
}

void FormulaUnittest::baseTest(){
  Formula formula;
  CPPUNIT_ASSERT_EQUAL(formula.getElementCount(),(unsigned int)0);
  CPPUNIT_ASSERT(formula.begin()==formula.end());

  // test some one letter element names
  formula += "H";
  CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)1);
  CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula.getElementCount(),(unsigned int)1);
  CPPUNIT_ASSERT(formula.begin()!=formula.end());
  formula += "H";
  CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)2);
  CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula.getElementCount(),(unsigned int)1);
  CPPUNIT_ASSERT(formula.begin()!=formula.end());
  formula += "O";
  CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)2);
  CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)1);
  CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula.getElementCount(),(unsigned int)2);
  CPPUNIT_ASSERT(formula.begin()!=formula.end());

  // test copy constructor
  Formula formula2=formula;
  CPPUNIT_ASSERT_EQUAL(formula2["H"],(unsigned int)2);
  CPPUNIT_ASSERT_EQUAL(formula2["O"],(unsigned int)1);
  CPPUNIT_ASSERT_EQUAL(formula2["C"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula2.getElementCount(),(unsigned int)2);
  CPPUNIT_ASSERT(formula2.begin()!=formula2.end());

  // test Assignment operator
  formula2=Formula();
  CPPUNIT_ASSERT_EQUAL(formula2["H"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula2["O"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula2["C"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula2.getElementCount(),(unsigned int)0);
  CPPUNIT_ASSERT(formula2.begin()==formula2.end());

  // test comparison
  CPPUNIT_ASSERT(formula!=formula2);
  Formula formula3;
  formula3 += "H";
  formula3 += "H";
  formula3 += "O";
  CPPUNIT_ASSERT(formula==formula3);

  // inequality
  formula3 += "O";
  CPPUNIT_ASSERT(formula!=formula3);

  // after removing it should be fine again
  formula3 -= "O";
  CPPUNIT_ASSERT(formula==formula3);

  // add something from the far end, to test resizing
  formula3 += "Rh";
  CPPUNIT_ASSERT(formula!=formula3);
  formula3 -= "Rh";
  CPPUNIT_ASSERT(formula==formula3);

  // removal of elements
  formula -= "H";
  CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)1);
  CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)1);
  CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula.getElementCount(),(unsigned int)2);
  CPPUNIT_ASSERT(formula.begin()!=formula.end());

  formula -= "O";
  CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)1);
  CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula.getElementCount(),(unsigned int)1);
  CPPUNIT_ASSERT(formula.begin()!=formula.end());

  formula -= "H";
  CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  CPPUNIT_ASSERT_EQUAL(formula.getElementCount(),(unsigned int)0);
  CPPUNIT_ASSERT(formula.begin()==formula.end());
}

void FormulaUnittest::parseTest(){
  {
    Formula formula("");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  }
  {
    Formula formula("H");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)1);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  }
  {
    Formula formula("H2");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)2);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  }
  {
    Formula formula("H2O");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)2);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)1);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  }
  {
    Formula formula("CH3COOH");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)4);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)2);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)2);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  }
  {
    Formula formula("CH3COONa");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)3);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)2);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)2);
    CPPUNIT_ASSERT_EQUAL(formula["Na"],(unsigned int)1);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  }
  {
    Formula formula("NaCH3COO");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)3);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)2);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)2);
    CPPUNIT_ASSERT_EQUAL(formula["Na"],(unsigned int)1);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  }
  {
    Formula formula("Na2CO3");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)3);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)1);
    CPPUNIT_ASSERT_EQUAL(formula["Na"],(unsigned int)2);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  }
  {
    Formula formula("CH2(COOH)2");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)4);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)4);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)3);
    CPPUNIT_ASSERT_EQUAL(formula["Na"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
  }
  {
    Formula formula("K4[Fe(CN)6]");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)6);
    CPPUNIT_ASSERT_EQUAL(formula["Na"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["K"],(unsigned int)4);
    CPPUNIT_ASSERT_EQUAL(formula["Fe"],(unsigned int)1);
    CPPUNIT_ASSERT_EQUAL(formula["N"],(unsigned int)6);
  }
  {
    Formula formula("[CrCl3(H2O)3]");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)6);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)3);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["Na"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["Cr"],(unsigned int)1);
    CPPUNIT_ASSERT_EQUAL(formula["Cl"],(unsigned int)3);
  }
  {
    Formula formula("Mg3[Fe(CN)6]2");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)12);
    CPPUNIT_ASSERT_EQUAL(formula["Na"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["Mg"],(unsigned int)3);
    CPPUNIT_ASSERT_EQUAL(formula["Fe"],(unsigned int)2);
    CPPUNIT_ASSERT_EQUAL(formula["N"],(unsigned int)12);
  }
  {
    Formula formula("Na[Fe((HO2CCH2)2NCH2CH2N(CH2CO2H)2)]");
    CPPUNIT_ASSERT_EQUAL(formula["H"],(unsigned int)16);
    CPPUNIT_ASSERT_EQUAL(formula["O"],(unsigned int)8);
    CPPUNIT_ASSERT_EQUAL(formula["C"],(unsigned int)10);
    CPPUNIT_ASSERT_EQUAL(formula["Na"],(unsigned int)1);
    CPPUNIT_ASSERT_EQUAL(formula["He"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["Mg"],(unsigned int)0);
    CPPUNIT_ASSERT_EQUAL(formula["Fe"],(unsigned int)1);
    CPPUNIT_ASSERT_EQUAL(formula["N"],(unsigned int)2);
  }
  {
    CPPUNIT_ASSERT_THROW(Formula formula("74107"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("  "),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("nAcL"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("NAcL"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("Na Cl"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("1NaCl"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("Mag"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("AgCl)"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("(Na"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("(Mag)"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("MgCl2)"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("((MgCl2)"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("(MgCl2))"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("(MgCl2]"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("[MgCl2)"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("N(aCl"),ParseError);
    CPPUNIT_ASSERT_THROW(Formula formula("Na()Cl"),ParseError);
  }


}
