/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/** \file element.cpp
 * 
 * Function implementations for the class element.
 * 
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include <iomanip>
#include <fstream>

#include "Helpers/Assert.hpp"
#include "element.hpp"

using namespace std;

/************************************* Functions for class element **********************************/

/** Constructor of class element.
 */
element::element() :
  mass(0),
  CovalentRadius(0),
  VanDerWaalsRadius(0),
	Z(-1),
	Valence(0),
	NoValenceOrbitals(0)
{
};

element::element(const element &src) :
  mass(src.mass),
  CovalentRadius(src.CovalentRadius),
  VanDerWaalsRadius(src.VanDerWaalsRadius),
  Z(src.Z),
  Valence(src.Valence),
  NoValenceOrbitals(src.NoValenceOrbitals),
  name(src.name),
  symbol(src.symbol)
{
  period = src.period;
  group = src.group;
  block = src.block;
}

/** Destructor of class element.
 */
element::~element() {};

element &element::operator=(const element &src){
  if(this!=&src){
    mass=src.mass;
    CovalentRadius=src.CovalentRadius;
    VanDerWaalsRadius=src.VanDerWaalsRadius;
    Z=src.Z;
    Valence=src.Valence;
    NoValenceOrbitals=src.NoValenceOrbitals;
    name=src.name;
    symbol=src.symbol;
    period = src.period;
    group = src.group;
    block = src.block;
  }
  return *this;
}

/** Prints element data to \a *out.
 * \param *out outstream
 */
bool element::Output(ostream * const out) const
{ 
  if (out != NULL) {
    *out << name << "\t" << symbol << "\t" << period << "\t" << group << "\t" << block << "\t" << Z << "\t" << mass << "\t" << CovalentRadius << "\t" << VanDerWaalsRadius << endl;
    //*out << Z << "\t"  << fixed << setprecision(11) << showpoint << mass << "g/mol\t" << name << "\t" << symbol << "\t" << endl;
    return true;
  } else
    return false;
};

/** Prints element data to \a *out.
 * \param *out outstream
 * \param No  cardinal number of element
 * \param NoOfAtoms total number of atom of this element type
 */
bool element::Checkout(ostream * const out, const int Number, const int NoOfAtoms) const
{ 
  if (out != NULL) {
    *out << "Ion_Type" << Number << "\t" << NoOfAtoms << "\t" << Z << "\t1.0\t3\t3\t" << fixed << setprecision(11) << showpoint << mass << "\t" << name << "\t" << symbol <<endl;
    return true;
  } else
    return false;
};

atomicNumber_t element::getNumber() const{
  return Z;
}

double element::getMass() const
{
  return mass;
}

double element::getCovalentRadius() const
{
  return CovalentRadius;
}

double element::getVanDerWaalsRadius() const
{
  return VanDerWaalsRadius;
}

int element::getAtomicNumber() const
{
  return Z;
}

double element::getValence() const
{
  return Valence;
}

int element::getNoValenceOrbitals() const
{
  return NoValenceOrbitals;
}

double element::getHBondDistance(const int i) const
{
  ASSERT((i>=0) && (i<3), "Access to element::HBondDistance out of bounds.");
  return HBondDistance[i];
}

double element::getHBondAngle(const int i) const
{
  ASSERT((i>=0) && (i<3), "Access to element::HBondAngle out of bounds.");
  return HBondAngle[i];
}

string &element::getSymbol(){
  return symbol;
}

const string &element::getSymbol() const{
  return symbol;
}

void element::setSymbol(const std::string &temp)
{
  symbol = temp;
}

std::string &element::getName(){
  return name;
}

const std::string &element::getName() const{
  return name;
}

void element::setName(const std::string &temp)
{
  name = temp;
}

std::ostream &operator<<(std::ostream &ost,const element &elem){
  ost << elem.getName() << "(" << elem.getNumber() << ")";
  return ost;
}
