/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * atom_bondedparticle.cpp
 *
 *  Created on: Oct 19, 2009
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "atom.hpp"
#include "atom_bondedparticle.hpp"
#include "bond.hpp"
#include "element.hpp"
#include "lists.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"

/** Constructor of class BondedParticle.
 */
BondedParticle::BondedParticle()
{
};

/** Destructor of class BondedParticle.
 */
BondedParticle::~BondedParticle()
{
  BondList::const_iterator Runner;
  while (!ListOfBonds.empty()) {
    Runner = ListOfBonds.begin();
    removewithoutcheck(*Runner);
  }
};

/** Outputs the current atom::AdaptiveOrder and atom::MaxOrder to \a *file.
 * \param *file output stream
 */
void BondedParticle::OutputOrder(ofstream *file) const
{
  *file << nr << "\t" << (int)AdaptiveOrder << "\t" << (int)MaxOrder << endl;
  //Log() << Verbose(2) << "Storing: " << nr << "\t" << (int)AdaptiveOrder << "\t" << (int)MaxOrder << "." << endl;
};

/** Prints all bonds of this atom with total degree.
 */
void BondedParticle::OutputBondOfAtom() const
{
  DoLog(4) && (Log() << Verbose(4) << "Atom " << getName() << "/" << nr << " with " << ListOfBonds.size() << " bonds: " << endl);
  int TotalDegree = 0;
  for (BondList::const_iterator Runner = ListOfBonds.begin(); Runner != ListOfBonds.end(); ++Runner) {
    DoLog(4) && (Log() << Verbose(4) << **Runner << endl);
    TotalDegree += (*Runner)->BondDegree;
  }
  DoLog(4) && (Log() << Verbose(4) << " -- TotalDegree: " << TotalDegree << endl);
};

/** Output of atom::nr along with all bond partners.
 * \param *AdjacencyFile output stream
 */
void BondedParticle::OutputAdjacency(ofstream * const AdjacencyFile) const
{
  *AdjacencyFile << nr << "\t";
  for (BondList::const_iterator Runner = ListOfBonds.begin(); Runner != ListOfBonds.end(); (++Runner))
    *AdjacencyFile << (*Runner)->GetOtherAtom(this)->nr << "\t";
  *AdjacencyFile << endl;
};

/** Output of atom::nr along each bond partner per line.
 * Only bonds are printed where atom::nr is smaller than the one of the bond partner.
 * \param *AdjacencyFile output stream
 */
void BondedParticle::OutputBonds(ofstream * const BondFile) const
{
  for (BondList::const_iterator Runner = ListOfBonds.begin(); Runner != ListOfBonds.end(); (++Runner))
    if (nr < (*Runner)->GetOtherAtom(this)->nr)
      *BondFile << nr << "\t" << (*Runner)->GetOtherAtom(this)->nr << "\n";
};

/**
 * Adds a bond between this bonded particle and another. Does nothing if this
 * bond already exists.
 *
 * \param bonding partner
 */
void BondedParticle::addBond(BondedParticle* Partner) {
  if (IsBondedTo(Partner)) {
    return;
  }

  bond* newBond = new bond((atom*) this, (atom*) Partner, 1, 0);
  RegisterBond(newBond);
  Partner->RegisterBond(newBond);
}

/** Puts a given bond into atom::ListOfBonds.
 * \param *Binder bond to insert
 */
bool BondedParticle::RegisterBond(bond *Binder)
{
  bool status = false;
  if (Binder != NULL) {
    if (Binder->Contains(this)) {
      ListOfBonds.push_back(Binder);
      status = true;
    } else {
      DoeLog(1) && (eLog()<< Verbose(1) << *Binder << " does not contain " << *this << "." << endl);
    }
  } else {
    DoeLog(1) && (eLog()<< Verbose(1) << "Binder is " << Binder << "." << endl);
  }
  return status;
};

/** Removes a given bond from atom::ListOfBonds.
 * \param *Binder bond to remove
 */
bool BondedParticle::UnregisterBond(bond *Binder)
{
  bool status = false;
  if (Binder != NULL) {
    if (Binder->Contains(this)) {
      ListOfBonds.remove(Binder);
      status = true;
    } else {
      DoeLog(1) && (eLog()<< Verbose(1) << *Binder << " does not contain " << *this << "." << endl);
    }
  } else {
    DoeLog(1) && (eLog()<< Verbose(1) << "Binder is " << Binder << "." << endl);
  }
  return status;
};

/** Removes all bonds from atom::ListOfBonds.
 * \note Does not do any memory de-allocation.
 */
void BondedParticle::UnregisterAllBond()
{
  ListOfBonds.clear();
};

/** Corrects the bond degree by one at most if necessary.
 * \param *out output stream for debugging
 */
int BondedParticle::CorrectBondDegree()
{
  int NoBonds = 0;
  int OtherNoBonds = 0;
  int FalseBondDegree = 0;
  atom *OtherWalker = NULL;
  bond *CandidateBond = NULL;

  NoBonds = CountBonds();
  //Log() << Verbose(3) << "Walker " << *this << ": " << (int)this->type->NoValenceOrbitals << " > " << NoBonds << "?" << endl;
  if ((int)(getType()->getNoValenceOrbitals()) > NoBonds) { // we have a mismatch, check all bonding partners for mismatch
    for (BondList::const_iterator Runner = ListOfBonds.begin(); Runner != ListOfBonds.end(); (++Runner)) {
      OtherWalker = (*Runner)->GetOtherAtom(this);
      OtherNoBonds = OtherWalker->CountBonds();
      //Log() << Verbose(3) << "OtherWalker " << *OtherWalker << ": " << (int)OtherWalker->type->NoValenceOrbitals << " > " << OtherNoBonds << "?" << endl;
      if ((int)(OtherWalker->getType()->getNoValenceOrbitals()) > OtherNoBonds) { // check if possible candidate
        if ((CandidateBond == NULL) || (ListOfBonds.size() > OtherWalker->ListOfBonds.size())) { // pick the one with fewer number of bonds first
          CandidateBond = (*Runner);
          //Log() << Verbose(3) << "New candidate is " << *CandidateBond << "." << endl;
        }
      }
    }
    if ((CandidateBond != NULL)) {
      CandidateBond->BondDegree++;
      //Log() << Verbose(2) << "Increased bond degree for bond " << *CandidateBond << "." << endl;
    } else {
      DoeLog(2) && (eLog()<< Verbose(2) << "Could not find correct degree for atom " << *this << "." << endl);
      FalseBondDegree++;
    }
  }
  return FalseBondDegree;
};

/** Checks whether there is a bond between \a this atom and the given \a *BondPartner.
 * \param *BondPartner atom to check for
 * \return true - bond exists, false - bond does not exist
 */
bool BondedParticle::IsBondedTo(BondedParticle * const BondPartner)
{
  bool status = false;

  for (BondList::iterator runner = ListOfBonds.begin(); runner != ListOfBonds.end(); runner++) {
    status = status || ((*runner)->Contains(BondPartner));
  }
  return status;
};

std::ostream & BondedParticle::operator << (std::ostream &ost) const
{
  ParticleInfo::operator<<(ost);
  ost << "," << getPosition();
  return ost;
}

std::ostream & operator << (std::ostream &ost, const BondedParticle &a)
{
  a.ParticleInfo::operator<<(ost);
  ost << "," << a.getPosition();
  return ost;
}

