/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * CommandLineDialog.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include <iostream>
#include <vector>

#include <Descriptors/AtomDescriptor.hpp>
#include <Descriptors/AtomIdDescriptor.hpp>
#include <Descriptors/MoleculeDescriptor.hpp>
#include <Descriptors/MoleculeIdDescriptor.hpp>
#include "CommandLineUI/CommandLineDialog.hpp"

#include "Actions/Values.hpp"

#include "element.hpp"
#include "periodentafel.hpp"
#include "CommandLineParser.hpp"
#include "defs.hpp"
#include "Helpers/Log.hpp"
#include "LinearAlgebra/Matrix.hpp"
#include "periodentafel.hpp"
#include "Helpers/Verbose.hpp"
#include "World.hpp"
#include "Box.hpp"

#include "atom.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "LinearAlgebra/Vector.hpp"

using namespace std;


CommandLineDialog::CommandLineDialog()
{
}

CommandLineDialog::~CommandLineDialog()
{
}


void CommandLineDialog::queryEmpty(const char* title, string _description){
  registerQuery(new EmptyCommandLineQuery(title, _description));
}

void CommandLineDialog::queryInt(const char* title, string _description){
  registerQuery(new IntCommandLineQuery(title, _description));
}

void CommandLineDialog::queryInts(const char* title, string _description){
  registerQuery(new IntsCommandLineQuery(title, _description));
}

void CommandLineDialog::queryBoolean(const char* title, string _description){
  registerQuery(new BooleanCommandLineQuery(title, _description));
}

void CommandLineDialog::queryDouble(const char* title, string _description){
  registerQuery(new DoubleCommandLineQuery(title, _description));
}

void CommandLineDialog::queryDoubles(const char* title, string _description){
  registerQuery(new DoublesCommandLineQuery(title, _description));
}

void CommandLineDialog::queryString(const char* title, string _description){
  registerQuery(new StringCommandLineQuery(title, _description));
}

void CommandLineDialog::queryStrings(const char* title, string _description){
  registerQuery(new StringsCommandLineQuery(title, _description));
}

void CommandLineDialog::queryAtom(const char* title, string _description) {
  registerQuery(new AtomCommandLineQuery(title, _description));
}

void CommandLineDialog::queryAtoms(const char* title, string _description) {
  registerQuery(new AtomsCommandLineQuery(title, _description));
}

void CommandLineDialog::queryMolecule(const char* title, string _description) {
  registerQuery(new MoleculeCommandLineQuery(title, _description));
}

void CommandLineDialog::queryMolecules(const char* title, string _description) {
  registerQuery(new MoleculesCommandLineQuery(title, _description));
}

void CommandLineDialog::queryVector(const char* title, bool check, string _description) {
  registerQuery(new VectorCommandLineQuery(title,check, _description));
}

void CommandLineDialog::queryVectors(const char* title, bool check, string _description) {
  registerQuery(new VectorsCommandLineQuery(title,check, _description));
}

void CommandLineDialog::queryBox(const char* title, string _description) {
  registerQuery(new BoxCommandLineQuery(title,_description));
}

void CommandLineDialog::queryElement(const char* title, string _description){
  registerQuery(new ElementCommandLineQuery(title, _description));
}

void CommandLineDialog::queryElements(const char* title, string _description){
  registerQuery(new ElementsCommandLineQuery(title, _description));
}

/************************** Query Infrastructure ************************/

CommandLineDialog::EmptyCommandLineQuery::EmptyCommandLineQuery(string title, string _description) :
    Dialog::EmptyQuery(title, _description)
{}

CommandLineDialog::EmptyCommandLineQuery::~EmptyCommandLineQuery() {}

bool CommandLineDialog::EmptyCommandLineQuery::handle() {
  cout << "Message of " << getTitle() << ":\n" << getDescription() << "\n";
  return true;
}

CommandLineDialog::IntCommandLineQuery::IntCommandLineQuery(string title, string _description) :
    Dialog::IntQuery(title, _description)
{}

CommandLineDialog::IntCommandLineQuery::~IntCommandLineQuery() {}

bool CommandLineDialog::IntCommandLineQuery::handle() {
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    tmp = CommandLineParser::getInstance().vm[getTitle()].as<int>();
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing integer for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::IntsCommandLineQuery::IntsCommandLineQuery(string title, string _description) :
    Dialog::IntsQuery(title, _description)
{}

CommandLineDialog::IntsCommandLineQuery::~IntsCommandLineQuery() {}

bool CommandLineDialog::IntsCommandLineQuery::handle() {
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    tmp = CommandLineParser::getInstance().vm[getTitle()].as< std::vector<int> >();
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing integers for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::BooleanCommandLineQuery::BooleanCommandLineQuery(string title, string _description) :
    Dialog::BooleanQuery(title, _description)
{}

CommandLineDialog::BooleanCommandLineQuery::~BooleanCommandLineQuery() {}

bool CommandLineDialog::BooleanCommandLineQuery::handle() {
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    tmp = CommandLineParser::getInstance().vm[getTitle()].as<bool>();
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing boolean for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::StringCommandLineQuery::StringCommandLineQuery(string title, string _description) :
    Dialog::StringQuery(title, _description)
{}

CommandLineDialog::StringCommandLineQuery::~StringCommandLineQuery() {}

bool CommandLineDialog::StringCommandLineQuery::handle() {
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    tmp = CommandLineParser::getInstance().vm[getTitle()].as<string>();
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing string for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::StringsCommandLineQuery::StringsCommandLineQuery(string title, string _description) :
    Dialog::StringsQuery(title, _description)
{}

CommandLineDialog::StringsCommandLineQuery::~StringsCommandLineQuery() {}

bool CommandLineDialog::StringsCommandLineQuery::handle() {
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    tmp = CommandLineParser::getInstance().vm[getTitle()].as< vector<string> >();
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing strings for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::DoubleCommandLineQuery::DoubleCommandLineQuery(string title, string _description) :
    Dialog::DoubleQuery(title, _description)
{}

CommandLineDialog::DoubleCommandLineQuery::~DoubleCommandLineQuery() {}

bool CommandLineDialog::DoubleCommandLineQuery::handle() {
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    tmp = CommandLineParser::getInstance().vm[getTitle()].as<double>();
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing double for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::DoublesCommandLineQuery::DoublesCommandLineQuery(string title, string _description) :
    Dialog::DoublesQuery(title, _description)
{}

CommandLineDialog::DoublesCommandLineQuery::~DoublesCommandLineQuery() {}

bool CommandLineDialog::DoublesCommandLineQuery::handle() {
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    tmp = CommandLineParser::getInstance().vm[getTitle()].as< std::vector<double> >();
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing doubles for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::AtomCommandLineQuery::AtomCommandLineQuery(string title, string _description) :
    Dialog::AtomQuery(title, _description)
{}

CommandLineDialog::AtomCommandLineQuery::~AtomCommandLineQuery() {}

bool CommandLineDialog::AtomCommandLineQuery::handle() {
  int IdxOfAtom = -1;
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    IdxOfAtom = CommandLineParser::getInstance().vm[getTitle()].as<int>();
    tmp = World::getInstance().getAtom(AtomById(IdxOfAtom));
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing atom for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::AtomsCommandLineQuery::AtomsCommandLineQuery(string title, string _description) :
    Dialog::AtomsQuery(title, _description)
{}

CommandLineDialog::AtomsCommandLineQuery::~AtomsCommandLineQuery() {}

bool CommandLineDialog::AtomsCommandLineQuery::handle() {
  std::vector<int> IdxOfAtom;
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    IdxOfAtom = CommandLineParser::getInstance().vm[getTitle()].as< std::vector<int> >();
    for (std::vector<int>::iterator iter = IdxOfAtom.begin(); iter != IdxOfAtom.end(); ++iter) {
      temp = World::getInstance().getAtom(AtomById(*iter));
      if (temp)
        tmp.push_back(temp);
    }
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing atoms for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::MoleculeCommandLineQuery::MoleculeCommandLineQuery(string title, string _description) :
    Dialog::MoleculeQuery(title, _description)
{}

CommandLineDialog::MoleculeCommandLineQuery::~MoleculeCommandLineQuery() {}

bool CommandLineDialog::MoleculeCommandLineQuery::handle() {
  int IdxOfMol = -1;
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    IdxOfMol = CommandLineParser::getInstance().vm[getTitle()].as<int>();
    tmp = World::getInstance().getMolecule(MoleculeById(IdxOfMol));
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing molecule for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::MoleculesCommandLineQuery::MoleculesCommandLineQuery(string title, string _description) :
    Dialog::MoleculesQuery(title, _description)
{}

CommandLineDialog::MoleculesCommandLineQuery::~MoleculesCommandLineQuery() {}

bool CommandLineDialog::MoleculesCommandLineQuery::handle() {
  std::vector<int> IdxOfMol;
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    IdxOfMol = CommandLineParser::getInstance().vm[getTitle()].as< std::vector<int> >();
    for (std::vector<int>::iterator iter = IdxOfMol.begin(); iter != IdxOfMol.end(); ++iter) {
      temp = World::getInstance().getMolecule(MoleculeById(*iter));
      if (temp)
        tmp.push_back(temp);
    }
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing molecules for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::VectorCommandLineQuery::VectorCommandLineQuery(string title, bool _check, string _description) :
    Dialog::VectorQuery(title,_check, _description)
{}

CommandLineDialog::VectorCommandLineQuery::~VectorCommandLineQuery()
{}

bool CommandLineDialog::VectorCommandLineQuery::handle() {
  VectorValue temp;
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    temp = CommandLineParser::getInstance().vm[getTitle()].as< VectorValue >();
    tmp[0] = temp.x;
    tmp[1] = temp.y;
    tmp[2] = temp.z;
    if ((check) && (!World::getInstance().getDomain().isInside(tmp))) {
      DoeLog(1) && (eLog() << Verbose(1) << "Vector " << tmp << " would be outside of box domain." << endl);
      return false;
    }
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing vector for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::VectorsCommandLineQuery::VectorsCommandLineQuery(string title, bool _check, string _description) :
    Dialog::VectorsQuery(title,_check, _description)
{}

CommandLineDialog::VectorsCommandLineQuery::~VectorsCommandLineQuery()
{}

bool CommandLineDialog::VectorsCommandLineQuery::handle() {
  std::vector<VectorValue> temporary;
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    temporary = CommandLineParser::getInstance().vm[getTitle()].as< std::vector<VectorValue> >();
    for(std::vector<VectorValue>::iterator iter = temporary.begin(); iter != temporary.end(); ++iter) {
      temp[0] = (*iter).x;
      temp[1] = (*iter).y;
      temp[2] = (*iter).z;
      if ((!check) || (World::getInstance().getDomain().isInside(temp)))
        tmp.push_back(temp);
      else
        DoeLog(1) && (eLog() << Verbose(1) << "Vector " << temp << " would be outside of box domain." << endl);
    }
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing vectors for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::BoxCommandLineQuery::BoxCommandLineQuery(string title, string _description) :
    Dialog::BoxQuery(title, _description)
{}

CommandLineDialog::BoxCommandLineQuery::~BoxCommandLineQuery()
{}

bool CommandLineDialog::BoxCommandLineQuery::handle() {
  BoxValue temp;
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    temp = CommandLineParser::getInstance().vm[getTitle()].as< BoxValue >();
    Matrix M;
    M.set(0,0, temp.xx);
    M.set(0,1, temp.yx);
    M.set(0,2, temp.zx);
    M.set(1,0, temp.yx);
    M.set(1,1, temp.yy);
    M.set(1,2, temp.zy);
    M.set(2,0, temp.zx);
    M.set(2,1, temp.zy);
    M.set(2,2, temp.zz);
    tmp.setM(M);
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing symmetric box matrix for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::ElementCommandLineQuery::ElementCommandLineQuery(string title, string _description) :
    Dialog::ElementQuery(title, _description)
{}

CommandLineDialog::ElementCommandLineQuery::~ElementCommandLineQuery()
{}

bool CommandLineDialog::ElementCommandLineQuery::handle() {
  // TODO: vector of ints and removing first is not correctly implemented yet. How to remove from a vector?
  periodentafel *periode = World::getInstance().getPeriode();
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    int Z = CommandLineParser::getInstance().vm[getTitle()].as< int >();
    tmp = periode->FindElement(Z);
    ASSERT(tmp != NULL, "Invalid element specified in ElementCommandLineQuery");
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing element for " << getTitle() << "." << endl);
    return false;
  }
}

CommandLineDialog::ElementsCommandLineQuery::ElementsCommandLineQuery(string title, string _description) :
    Dialog::ElementsQuery(title, _description)
{}

CommandLineDialog::ElementsCommandLineQuery::~ElementsCommandLineQuery()
{}

bool CommandLineDialog::ElementsCommandLineQuery::handle() {
  // TODO: vector of ints and removing first is not correctly implemented yet. How to remove from a vector?
  periodentafel *periode = World::getInstance().getPeriode();
  if (CommandLineParser::getInstance().vm.count(getTitle())) {
    vector<int> AllElements = CommandLineParser::getInstance().vm[getTitle()].as< vector<int> >();
    for (vector<int>::iterator ZRunner = AllElements.begin(); ZRunner != AllElements.end(); ++ZRunner) {
      temp = periode->FindElement(*ZRunner);
      ASSERT(temp != NULL, "Invalid element specified in ElementCommandLineQuery");
      tmp.push_back(temp);
    }
    return true;
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "CommandLineUI parsing error: Missing elements for " << getTitle() << "." << endl);
    return false;
  }
}
