/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * BaseShapes_impl.cpp
 *
 *  Created on: Jun 18, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Shapes/BaseShapes.hpp"
#include "Shapes/BaseShapes_impl.hpp"

#include <cmath>

#include "Helpers/Assert.hpp"
#include "LinearAlgebra/Vector.hpp"

bool Sphere_impl::isInside(const Vector &point){
  return point.NormSquared()<=1;
}


/**
 * algorithm taken from http://www.cgafaq.info/wiki/Evenly_distributed_points_on_sphere
 * \param N number of points on surface
 */
std::vector<Vector> Sphere_impl::getHomogeneousPointsOnSurface(const size_t N) const {
  std::vector<Vector> PointsOnSurface;

  const double dlength = M_PI*(3.-sqrt(5.));
  double length = 0;
  const double dz = 2.0/N;
  double z = 1. - dz/2.;
  Vector point;
  for (size_t ka = 0; ka<N; ka++){
    const double r = sqrt(1.-z*z);
    point.Zero();
    point[0] = cos(length)*r;
    point[1] = sin(length)*r;
    point[2] = z;
    PointsOnSurface.push_back(point);
    z = z - dz;
    length = length + dlength;
  }

  ASSERT(PointsOnSurface.size() == N, "Sphere_impl::getHomogeneousPointsOnSurface() did not create enough points.");
  return PointsOnSurface;
}


Shape Sphere(){
  Shape::impl_ptr impl = Shape::impl_ptr(new Sphere_impl());
  return Shape(impl);
}

bool Cuboid_impl::isInside(const Vector &point){
  return (point[0]>=0 && point[0]<=1) && (point[1]>=0 && point[1]<=1) && (point[2]>=0 && point[2]<=1);
}

/**
 * \param N number of points on surface
 */
std::vector<Vector> Cuboid_impl::getHomogeneousPointsOnSurface(const size_t N) const {
  std::vector<Vector> PointsOnSurface;
  ASSERT(false, "Cuboid_impl::getHomogeneousPointsOnSurface() not implemented yet");
  return PointsOnSurface;
}

Shape Cuboid(){
  Shape::impl_ptr impl = Shape::impl_ptr(new Cuboid_impl());
  return Shape(impl);
}
