/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/** \file helpers.cpp
 *
 * Implementation of some auxiliary functions for memory dis-/allocation and so on
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Helpers/helpers.hpp"
#include "Helpers/fast_functions.hpp"
#include "Helpers/Verbose.hpp"
#include "Helpers/Log.hpp"

#include <iostream>

/********************************************** helpful functions *********************************/


/** Asks for a double value and checks input
 * \param *text question
 */
double ask_value(const char *text)
{
  double test = 0.1439851348959832147598734598273456723948652983045928346598365;
  do {
    DoLog(0) && (Log() << Verbose(0) << text);
    cin >> test;
  } while (test == 0.1439851348959832147598734598273456723948652983045928346598365);
  return test;
};

/** Output of a debug message to stderr.
 * \param *P Problem at hand, points to ParallelSimulationData#me
 * \param output output string
 */
#ifdef HAVE_DEBUG
void debug_in(const char *output, const char *file, const int line) {
  if (output) fprintf(stderr,"DEBUG: in %s at line %i: %s\n", file, line, output);
}
#else
void debug_in(const char *output, const char *file, const int line) {}  // print nothing
#endif

/** modulo operator for doubles.
 * \param *b pointer to double
 * \param lower_bound lower bound
 * \param upper_bound upper bound
 */
void bound(double *b, double lower_bound, double upper_bound)
{
  double step = (upper_bound - lower_bound);
  while (*b >= upper_bound)
    *b -= step;
  while (*b < lower_bound)
    *b += step;
};

/** Counts lines in file.
 * Note we are scanning lines from current position, not from beginning.
 * \param InputFile file to be scanned.
 */
int CountLinesinFile(ifstream &InputFile)
{
  char *buffer = new char[MAXSTRINGSIZE];
  int lines=0;

  int PositionMarker = InputFile.tellg();  // not needed as Inputfile is copied, given by value, not by ref
  // count the number of lines, i.e. the number of fragments
  InputFile.getline(buffer, MAXSTRINGSIZE); // skip comment lines
  InputFile.getline(buffer, MAXSTRINGSIZE);
  while(!InputFile.eof()) {
    InputFile.getline(buffer, MAXSTRINGSIZE);
    lines++;
  }
  InputFile.seekg(PositionMarker, ios::beg);
  delete[](buffer);
  return lines;
};

/** Returns a string with \a i prefixed with 0s to match order of total number of molecules in digits.
 * \param FragmentNumber total number of fragments to determine necessary number of digits
 * \param digits number to create with 0 prefixed
 * \return allocated(!) char array with number in digits, ten base.
 */
char *FixedDigitNumber(const int FragmentNumber, const int digits)
{
  char *returnstring;
  int number = FragmentNumber;
  int order = 0;
  while (number != 0) { // determine number of digits needed
    number = (int)floor(((double)number / 10.));
    order++;
    //Log() << Verbose(0) << "Number is " << number << ", order is " << order << "." << endl;
  }
  // allocate string
  returnstring = new char[order + 2];
  // terminate  and fill string array from end backward
  returnstring[order] = '\0';
  number = digits;
  for (int i=order;i--;) {
    returnstring[i] = '0' + (char)(number % 10);
    number = (int)floor(((double)number / 10.));
  }
  //Log() << Verbose(0) << returnstring << endl;
  return returnstring;
};

/** Tests whether a given string contains a valid number or not.
 * \param *string
 * \return true - is a number, false - is not a valid number
 */
bool IsValidNumber( const char *string)
{
  int ptr = 0;
  if ((string[ptr] == '.') || (string[ptr] == '-')) // number may be negative or start with dot
    ptr++;
  if ((string[ptr] >= '0') && (string[ptr] <= '9'))
    return true;
  return false;
};

/** Comparison function for GSL heapsort on distances in two molecules.
 * \param *a
 * \param *b
 * \return <0, \a *a less than \a *b, ==0 if equal, >0 \a *a greater than \a *b
 */
int CompareDoubles (const void * a, const void * b)
{
  if (*(double *)a > *(double *)b)
    return -1;
  else if (*(double *)a < *(double *)b)
    return 1;
  else
    return 0;
};


/**
 * Calls exit(255).
 */
void performCriticalExit() {
  exit(255);
}
