/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * MoleculeDescriptor.cpp
 *
 *  Created on: Feb 5, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Descriptors/MoleculeDescriptor.hpp"
#include "Descriptors/MoleculeDescriptor_impl.hpp"

#include "World.hpp"
#include "Patterns/ObservedContainer_impl.hpp"

#include "molecule.hpp"

#include <boost/bind.hpp>
#include <iostream>

using namespace std;

typedef World::MoleculeSet::internal_iterator molecules_iter_t;

/************************ Forwarding object **************************************/


MoleculeDescriptor::MoleculeDescriptor(impl_ptr _impl) :
    impl(_impl)
{}

MoleculeDescriptor::MoleculeDescriptor(const MoleculeDescriptor& src) :
    impl(src.get_impl())
{}

MoleculeDescriptor::~MoleculeDescriptor()
{}

MoleculeDescriptor& MoleculeDescriptor::operator=(MoleculeDescriptor &src){
  if(&src!=this) {
    impl=src.get_impl();
  }
  return *this;
}

molecule* MoleculeDescriptor::find(){
  return impl->find();
}

std::vector<molecule*> MoleculeDescriptor::findAll(){
  return impl->findAll();
}

MoleculeDescriptor::impl_ptr MoleculeDescriptor::get_impl() const{
  return impl;
}




/**************************** implementation ********************/

MoleculeDescriptor_impl::MoleculeDescriptor_impl()
{
}

MoleculeDescriptor_impl::~MoleculeDescriptor_impl()
{
}

World::MoleculeSet& MoleculeDescriptor_impl::getMolecules(){
  return World::getInstance().molecules;
}

molecule* MoleculeDescriptor_impl::find() {
  World::MoleculeSet &molecules = getMolecules();
  molecules_iter_t res = find_if(molecules.begin_internal(),molecules.end_internal(),boost::bind(&MoleculeDescriptor_impl::predicate,this,_1));
  return (res!=molecules.end_internal())?((*res).second):0;
}

vector<molecule*> MoleculeDescriptor_impl::findAll() {
  vector<molecule*> res;
  World::MoleculeSet &molecules = getMolecules();
  for_each(molecules.begin_internal(),
           molecules.end_internal(),
           boost::bind(&MoleculeDescriptor_impl::checkAndAdd,
                       this,&res,_1));
  return res;
}

void MoleculeDescriptor_impl::checkAndAdd(std::vector<molecule*> *v,std::pair<moleculeId_t,molecule*> p){
  if(predicate(p)){
    v->push_back(p.second);
  }
}

/************************** Universe and Emptyset *****************/

MoleculeAllDescriptor_impl::MoleculeAllDescriptor_impl()
{}

MoleculeAllDescriptor_impl::~MoleculeAllDescriptor_impl()
{}

bool MoleculeAllDescriptor_impl::predicate(std::pair<moleculeId_t,molecule*>){
  return true;
}

MoleculeDescriptor AllMolecules(){
  return MoleculeDescriptor(MoleculeDescriptor::impl_ptr(new MoleculeAllDescriptor_impl));
}

MoleculeNoneDescriptor_impl::MoleculeNoneDescriptor_impl()
{}

MoleculeNoneDescriptor_impl::~MoleculeNoneDescriptor_impl()
{}

bool MoleculeNoneDescriptor_impl::predicate(std::pair<moleculeId_t,molecule*>){
  return false;
}

MoleculeDescriptor NoMolecules(){
  return MoleculeDescriptor(MoleculeDescriptor::impl_ptr(new MoleculeNoneDescriptor_impl));
}

/************************** Operator stuff ************************/

// AND
MoleculeAndDescriptor_impl::MoleculeAndDescriptor_impl(MoleculeDescriptor::impl_ptr _lhs, MoleculeDescriptor::impl_ptr _rhs) :
    lhs(_lhs), rhs(_rhs)
{}

MoleculeAndDescriptor_impl::~MoleculeAndDescriptor_impl()
{}

bool MoleculeAndDescriptor_impl::predicate(std::pair<moleculeId_t,molecule*> molecule){
  return lhs->predicate(molecule) && rhs->predicate(molecule);
}
MoleculeDescriptor operator&&(const MoleculeDescriptor &lhs, const MoleculeDescriptor &rhs){
  MoleculeDescriptor::impl_ptr newImpl = MoleculeDescriptor::impl_ptr(new MoleculeAndDescriptor_impl(lhs.get_impl(),rhs.get_impl()));
  return MoleculeDescriptor(newImpl);
}

// OR
MoleculeOrDescriptor_impl::MoleculeOrDescriptor_impl(MoleculeDescriptor::impl_ptr _lhs ,MoleculeDescriptor::impl_ptr _rhs) :
    lhs(_lhs), rhs(_rhs)
{}

MoleculeOrDescriptor_impl::~MoleculeOrDescriptor_impl(){
}

bool MoleculeOrDescriptor_impl::predicate(std::pair<moleculeId_t,molecule*> molecule){
  return lhs->predicate(molecule) || rhs->predicate(molecule);
}

MoleculeDescriptor  operator||(const MoleculeDescriptor &lhs, const MoleculeDescriptor &rhs){
  MoleculeDescriptor::impl_ptr newImpl = MoleculeDescriptor::impl_ptr(new MoleculeOrDescriptor_impl(lhs.get_impl(),rhs.get_impl()));
  return MoleculeDescriptor(newImpl);
}

// NOT

MoleculeNotDescriptor_impl::MoleculeNotDescriptor_impl(MoleculeDescriptor::impl_ptr _arg) :
    arg(_arg)
{}


MoleculeNotDescriptor_impl::~MoleculeNotDescriptor_impl()
{
}

bool MoleculeNotDescriptor_impl::predicate(std::pair<moleculeId_t,molecule*> molecule){
 return !(arg->predicate(molecule));
}

MoleculeDescriptor operator!(const MoleculeDescriptor &arg){
  MoleculeDescriptor::impl_ptr newImpl = MoleculeDescriptor::impl_ptr(new MoleculeNotDescriptor_impl(arg.get_impl()));
  return MoleculeDescriptor(newImpl);
}
