/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ConfigFileBuffer.cpp
 *
 *  Created on: 12.06.2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "ConfigFileBuffer.hpp"
#include "Helpers/helpers.hpp"
#include "lists.hpp"
#include "Helpers/Verbose.hpp"
#include "Helpers/Log.hpp"
#include "World.hpp"

/******************************** Functions for class ConfigFileBuffer **********************/

/** Structure containing compare function for Ion_Type sorting.
 */
struct IonTypeCompare {
  bool operator()(const char* s1, const char *s2) const {
    char number1[8];
    char number2[8];
    const char *dummy1, *dummy2;
    //Log() << Verbose(0) << s1 << "  " << s2 << endl;
    dummy1 = strchr(s1, '_')+sizeof(char)*5;  // go just after "Ion_Type"
    dummy2 = strchr(dummy1, '_');
    strncpy(number1, dummy1, dummy2-dummy1); // copy the number
    number1[dummy2-dummy1]='\0';
    dummy1 = strchr(s2, '_')+sizeof(char)*5;  // go just after "Ion_Type"
    dummy2 = strchr(dummy1, '_');
    strncpy(number2, dummy1, dummy2-dummy1); // copy the number
    number2[dummy2-dummy1]='\0';
    if (atoi(number1) != atoi(number2))
      return (atoi(number1) < atoi(number2));
    else {
      dummy1 = strchr(s1, '_')+sizeof(char);
      dummy1 = strchr(dummy1, '_')+sizeof(char);
      dummy2 = strchr(dummy1, ' ') < strchr(dummy1, '\t') ? strchr(dummy1, ' ') : strchr(dummy1, '\t');
      strncpy(number1, dummy1, dummy2-dummy1); // copy the number
      number1[dummy2-dummy1]='\0';
      dummy1 = strchr(s2, '_')+sizeof(char);
      dummy1 = strchr(dummy1, '_')+sizeof(char);
      dummy2 = strchr(dummy1, ' ') < strchr(dummy1, '\t') ? strchr(dummy1, ' ') : strchr(dummy1, '\t');
      strncpy(number2, dummy1, dummy2-dummy1); // copy the number
      number2[dummy2-dummy1]='\0';
      return (atoi(number1) < atoi(number2));
    }
  }
};

/** Constructor for ConfigFileBuffer class.
 */
ConfigFileBuffer::ConfigFileBuffer() :
    buffer(NULL),
    LineMapping(NULL),
    CurrentLine(0),
    NoLines(0)
{
};

/** Constructor for ConfigFileBuffer class with filename to be parsed.
 * \param *filename file name
 */
ConfigFileBuffer::ConfigFileBuffer(const char * const filename) :
    buffer(NULL),
    LineMapping(NULL),
    CurrentLine(0),
    NoLines(0)
{
  InitFileBuffer(filename);
}

void ConfigFileBuffer::InitFileBuffer(const char * const filename)
{
  ifstream *file= new ifstream(filename);
  InitFileBuffer(file);
}

void ConfigFileBuffer::InitFileBuffer(istream *file)
{
  char line[MAXSTRINGSIZE];

  RemoveMapping();

  // prescan number of lines
  if (file->fail()) {
    DoeLog(1) && (eLog()<< Verbose(1) << "config file missing!" << endl);
    return;
  }
  NoLines = 0; // we're overcounting by one
  long file_position = file->tellg(); // mark current position
  do {
    file->getline(line, MAXSTRINGSIZE-1);
    NoLines++;
  } while (!file->eof());
  file->clear();
  file->seekg(file_position, ios::beg);
  DoLog(1) && (Log() << Verbose(1) << NoLines-1 << " lines were recognized." << endl);

  // allocate buffer's 1st dimension
  if (buffer != NULL) {
    DoeLog(1) && (eLog()<< Verbose(1) << "FileBuffer->buffer is not NULL!" << endl);
    return;
  } else
    buffer = new char *[NoLines];

  // scan each line and put into buffer
  int lines=0;
  int i;
  do {
    buffer[lines] = new char[MAXSTRINGSIZE];
    file->getline(buffer[lines], MAXSTRINGSIZE-1);
    i = strlen(buffer[lines]);
    buffer[lines][i] = '\n';
    buffer[lines][i+1] = '\0';
    lines++;
  } while((!file->eof()) && (lines < NoLines));
  DoLog(1) && (Log() << Verbose(1) << lines-1 << " lines were read into the buffer." << endl);
  file->clear();
  file->seekg(file_position, ios::beg);

  InitMapping();
}

/** Destructor for ConfigFileBuffer class.
 */
ConfigFileBuffer::~ConfigFileBuffer()
{
  RemoveBuffer();
  RemoveMapping();
}


/** Create trivial mapping.
 */
void ConfigFileBuffer::InitMapping()
{
  LineMapping = new int[NoLines];
  for (int i=0;i<NoLines;i++)
    LineMapping[i] = i;
  MappingAllocated = true;
}

/** Remove allocated mapping.
 */
void ConfigFileBuffer::RemoveMapping()
{
  delete[](LineMapping);
  MappingAllocated = false;
}

/** Remove allocated mapping.
 */
void ConfigFileBuffer::RemoveBuffer()
{
  for(int i=0;i<NoLines;++i)
    delete[](buffer[i]);
  delete[](buffer);
}


/** Creates a mapping for the \a *FileBuffer's lines containing the Ion_Type keyword such that they are sorted.
 * \a *map on return contains a list of NoAtom entries such that going through the list, yields indices to the
 * lines in \a *FileBuffer in a sorted manner of the Ion_Type?_? keywords. We assume that ConfigFileBuffer::CurrentLine
 * points to first Ion_Type entry.
 * \param *FileBuffer pointer to buffer structure
 * \param NoAtoms of subsequent lines to look at
 */
void ConfigFileBuffer::MapIonTypesInBuffer(const int NoAtoms)
{
  map<const char *, int, IonTypeCompare> IonTypeLineMap;
  if (!MappingAllocated) {
    InitMapping();
  }

  // put all into hashed map
  for (int i=0; i<NoAtoms; ++i) {
    IonTypeLineMap.insert(pair<const char *, int> (buffer[CurrentLine+i], CurrentLine+i));
  }

  // fill map
  int nr=0;
  for (map<const char *, int, IonTypeCompare>::iterator runner = IonTypeLineMap.begin(); runner != IonTypeLineMap.end(); ++runner) {
    if (CurrentLine+nr < NoLines)
      LineMapping[CurrentLine+(nr++)] = runner->second;
    else {
      DoeLog(0) && (eLog()<< Verbose(0) << "config::MapIonTypesInBuffer - NoAtoms is wrong: We are past the end of the file!" << endl);
      performCriticalExit();
    }
  }
}
