/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * BoundaryPointSet.cpp
 *
 *  Created on: Jul 29, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "BoundaryPointSet.hpp"

#include <iostream>

#include "BoundaryLineSet.hpp"
#include "BoundaryTriangleSet.hpp"
#include "TesselPoint.hpp"

#include "Helpers/Assert.hpp"
#include "Helpers/Info.hpp"
#include "Helpers/Log.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "Helpers/Verbose.hpp"

using namespace std;

// ======================================== Points on Boundary =================================

/** Constructor of BoundaryPointSet.
 */
BoundaryPointSet::BoundaryPointSet() :
  LinesCount(0),
  value(0.),
  Nr(-1)
{
  Info FunctionInfo(__func__);
  DoLog(1) && (Log() << Verbose(1) << "Adding noname." << endl);
}
;

/** Constructor of BoundaryPointSet with Tesselpoint.
 * \param *Walker TesselPoint this boundary point represents
 */
BoundaryPointSet::BoundaryPointSet(TesselPoint * const Walker) :
  LinesCount(0),
  node(Walker),
  value(0.),
  Nr(Walker->nr)
{
  Info FunctionInfo(__func__);
  DoLog(1) && (Log() << Verbose(1) << "Adding Node " << *Walker << endl);
}
;

/** Destructor of BoundaryPointSet.
 * Sets node to NULL to avoid removing the original, represented TesselPoint.
 * \note When removing point from a class Tesselation, use RemoveTesselationPoint()
 */
BoundaryPointSet::~BoundaryPointSet()
{
  Info FunctionInfo(__func__);
  //Log() << Verbose(0) << "Erasing point nr. " << Nr << "." << endl;
  if (!lines.empty())
    DoeLog(2) && (eLog() << Verbose(2) << "Memory Leak! I " << *this << " am still connected to some lines." << endl);
  node = NULL;
}
;

/** Add a line to the LineMap of this point.
 * \param *line line to add
 */
void BoundaryPointSet::AddLine(BoundaryLineSet * const line)
{
  Info FunctionInfo(__func__);
  DoLog(1) && (Log() << Verbose(1) << "Adding " << *this << " to line " << *line << "." << endl);
  if (line->endpoints[0] == this) {
    lines.insert(LinePair(line->endpoints[1]->Nr, line));
  } else {
    lines.insert(LinePair(line->endpoints[0]->Nr, line));
  }
  LinesCount++;
}
;

/** output operator for BoundaryPointSet.
 * \param &ost output stream
 * \param &a boundary point
 */
ostream & operator <<(ostream &ost, const BoundaryPointSet &a)
{
  ost << "[" << a.Nr << "|" << a.node->getName() << " at " << a.node->getPosition() << "]";
  return ost;
}
;
