/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * NotAllAtomsInsideCuboidAction.cpp
 *
 *  Created on: Aug 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Descriptors/AtomDescriptor.hpp"
#include "Descriptors/AtomShapeDescriptor.hpp"
#include "atom.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"
#include "LinearAlgebra/Matrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "Shapes/BaseShapes.hpp"
#include "Shapes/Shape.hpp"
#include "Shapes/ShapeOps.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

using namespace std;

#include "Actions/SelectionAction/NotAllAtomsInsideCuboidAction.hpp"

// and construct the stuff
#include "NotAllAtomsInsideCuboidAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr SelectionNotAllAtomsInsideCuboidAction::performCall() {
  std::vector<atom *> selectedAtoms = World::getInstance().getSelectedAtoms();
  Matrix RotationMatrix;

  // obtain information
  getParametersfromValueStorage();
  RotationMatrix.rotation(params.Xangle, params.Yangle, params.Zangle);

  DoLog(1) && (Log() << Verbose(1) << "Unselecting all atoms inside a rotated " << RotationMatrix << " cuboid at " << params.position << " and extension of " << params.extension << "." << endl);
  Shape s = translate(transform(stretch(Cuboid(),params.extension),RotationMatrix),params.position);

  World::getInstance().unselectAllAtoms(AtomByShape(s));
  return Action::state_ptr(new SelectionNotAllAtomsInsideCuboidState(selectedAtoms, params));
}

Action::state_ptr SelectionNotAllAtomsInsideCuboidAction::performUndo(Action::state_ptr _state) {
  SelectionNotAllAtomsInsideCuboidState *state = assert_cast<SelectionNotAllAtomsInsideCuboidState*>(_state.get());

  World::getInstance().clearAtomSelection();
  for(std::vector<atom *>::iterator iter = state->selectedAtoms.begin(); iter != state->selectedAtoms.end(); ++iter)
    World::getInstance().selectAtom(*iter);

  return Action::state_ptr(_state);
}

Action::state_ptr SelectionNotAllAtomsInsideCuboidAction::performRedo(Action::state_ptr _state){
  SelectionNotAllAtomsInsideCuboidState *state = assert_cast<SelectionNotAllAtomsInsideCuboidState*>(_state.get());
  Matrix RotationMatrix;

  RotationMatrix.rotation(state->params.Xangle, state->params.Yangle, state->params.Zangle);
  Shape s = translate(transform(stretch(Cuboid(),state->params.extension),RotationMatrix),state->params.position);
  World::getInstance().unselectAllAtoms(AtomByShape(s));

  return Action::state_ptr(_state);
}

bool SelectionNotAllAtomsInsideCuboidAction::canUndo() {
  return true;
}

bool SelectionNotAllAtomsInsideCuboidAction::shouldUndo() {
  return true;
}

const string SelectionNotAllAtomsInsideCuboidAction::getName() {
  return NAME;
}
/** =========== end of function ====================== */
