/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * RotateAroundSelfByAngleAction.cpp
 *
 *  Created on: Aug 06, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"
#include "LinearAlgebra/Line.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "atom.hpp"
#include "molecule.hpp"


#include <iostream>
#include <fstream>
#include <string>

using namespace std;

#include "Actions/MoleculeAction/RotateAroundSelfByAngleAction.hpp"

// and construct the stuff
#include "RotateAroundSelfByAngleAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr MoleculeRotateAroundSelfByAngleAction::performCall() {
  molecule *mol = NULL;

  // obtain information
  getParametersfromValueStorage();

  // check whether a single atom and molecule is selected
  if (World::getInstance().getSelectedMolecules().size() != 1)
    return Action::failure;
  mol = World::getInstance().beginMoleculeSelection()->second;

  // check whether Axis is valid
  if (params.Axis.IsZero())
    return Action::failure;

  // convert from degrees to radian
  params.angle *= M_PI/180.;

  // Creation Line that is the rotation axis
  Vector *CenterOfGravity = mol->DetermineCenterOfGravity();
  Line RotationAxis(*CenterOfGravity, params.Axis);
  delete(CenterOfGravity);
  DoLog(0) && (Log() << Verbose(0) << "Rotate around self by " << params.angle << " along " << RotationAxis << "." << endl);

  for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
    (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), params.angle));
  }
  DoLog(0) && (Log() << Verbose(0) << "done." << endl);

  return Action::state_ptr(new MoleculeRotateAroundSelfByAngleState(mol, params));
}

Action::state_ptr MoleculeRotateAroundSelfByAngleAction::performUndo(Action::state_ptr _state) {
  MoleculeRotateAroundSelfByAngleState *state = assert_cast<MoleculeRotateAroundSelfByAngleState*>(_state.get());

  Vector *CenterOfGravity = state->mol->DetermineCenterOfGravity();
  Line RotationAxis(*CenterOfGravity, state->params.Axis);
  delete(CenterOfGravity);

  for (molecule::iterator iter = state->mol->begin(); iter != state->mol->end(); ++iter) {
    (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), -state->params.angle));
  }

  return Action::state_ptr(_state);
}

Action::state_ptr MoleculeRotateAroundSelfByAngleAction::performRedo(Action::state_ptr _state){
  MoleculeRotateAroundSelfByAngleState *state = assert_cast<MoleculeRotateAroundSelfByAngleState*>(_state.get());

  Vector *CenterOfGravity = state->mol->DetermineCenterOfGravity();
  Line RotationAxis(*CenterOfGravity, state->params.Axis);
  delete(CenterOfGravity);

  for (molecule::iterator iter = state->mol->begin(); iter != state->mol->end(); ++iter) {
    (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), state->params.angle));
  }

  return Action::state_ptr(_state);
}

bool MoleculeRotateAroundSelfByAngleAction::canUndo() {
  return true;
}

bool MoleculeRotateAroundSelfByAngleAction::shouldUndo() {
  return true;
}

const string MoleculeRotateAroundSelfByAngleAction::getName() {
  return NAME;
}
/** =========== end of function ====================== */
