/** \file parser.hpp
 *
 * Definitions of various class functions for the parsing of value files.
 *
 */


#ifndef PARSER_HPP_
#define PARSER_HPP_

/*********************************************** includes ***********************************/

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/****************************************** forward declarations *****************************/

class EnergyMatrix;
class ForceMatrix;
class HessianMatrix;
class KeySetsContainer;
class MatrixContainer;

/********************************************** definitions *********************************/

#define EnergySuffix ".energy.all"
#define EnergyFragmentSuffix ".energyfragment.all"
#define ForcesSuffix ".forces.all"
#define ForceFragmentSuffix ".forcefragment.all"
#define HcorrectionSuffix ".Hcorrection.all"
#define HcorrectionFragmentSuffix ".Hcorrectionfragment.all"
#define HessianSuffix ".hessian_xx.all"
#define HessianFragmentSuffix ".hessianfragment_xx.all"
#define OrderSuffix ".Order"
#define ShieldingSuffix ".sigma_all.csv"
#define ShieldingPASSuffix ".sigma_all_PAS.csv"
#define ShieldingFragmentSuffix ".sigma_all_fragment.all"
#define ShieldingPASFragmentSuffix ".sigma_all_PAS_fragment.all"
#define ChiSuffix ".chi_all.csv"
#define ChiPASSuffix ".chi_all_PAS.csv"
#define ChiFragmentSuffix ".chi_all_fragment.all"
#define ChiPASFragmentSuffix ".chi_all_PAS_fragment.all"
#define TimeSuffix ".speed"

// ======================================= FUNCTIONS ==========================================

bool FilePresent(const char *filename, bool test);
bool TestParams(int argc, char **argv);

// ======================================= CLASS KeySetsContainer =============================

class KeySetsContainer {
  public:
    int **KeySets;
    int *AtomCounter;
    int FragmentCounter;
    int Order;
    int *FragmentsPerOrder;
    int **OrderSet;

  KeySetsContainer();
  ~KeySetsContainer();

  bool ParseKeySets(const char *name, const int *ACounter, const int FCounter);
  bool ParseManyBodyTerms();
  bool Contains(const int GreaterSet, const int SmallerSet);
};

// ======================================= CLASS MatrixContainer =============================

class MatrixContainer {
  public:
    double ***Matrix;
    int **Indices;
    char **Header;
    int MatrixCounter;
    int *RowCounter;
    int *ColumnCounter;

  MatrixContainer();
  virtual ~MatrixContainer();
  
  bool InitialiseIndices(class MatrixContainer *Matrix = NULL);
  bool ParseMatrix(std::istream &input, int skiplines, int skipcolumns, int MatrixNr);
  virtual bool ParseFragmentMatrix(const char *name, const char *prefix, std::string suffix, int skiplines, int skipcolumns);
  bool AllocateMatrix(char **GivenHeader, int MCounter, int *RCounter, int *CCounter);
  bool ResetMatrix();
  double FindMinValue();
  double FindMaxValue();
  bool SetLastMatrix(double value, int skipcolumns);
  bool SetLastMatrix(double **values, int skipcolumns);
  //bool ParseIndices();
  //bool SumSubValues();
  bool SumSubManyBodyTerms(class MatrixContainer &Matrix, class KeySetsContainer &KeySets, int Order);
  bool WriteTotalFragments(const char *name, const char *prefix);
  bool WriteLastMatrix(const char *name, const char *prefix, const char *suffix);
};

std::ostream & operator << (std::ostream &ost, const MatrixContainer &m);

// ======================================= CLASS EnergyMatrix =============================

class EnergyMatrix : public MatrixContainer {
  public:
    bool ParseIndices();
    bool SumSubEnergy(class EnergyMatrix &Fragments, class EnergyMatrix *CorrectionFragments, class KeySetsContainer &KeySets, int Order, double sign);
    bool ParseFragmentMatrix(const char *name, const char *prefix, std::string suffix, int skiplines, int skipcolumns);
};

// ======================================= CLASS ForceMatrix =============================

class ForceMatrix : public MatrixContainer {
  public:
    bool ParseIndices(const char *name);
    bool SumSubForces(class ForceMatrix &Fragments, class KeySetsContainer &KeySets, int Order, double sign);
    bool ParseFragmentMatrix(const char *name, const char *prefix, std::string suffix, int skiplines, int skipcolumns);
};

// ======================================= CLASS HessianMatrix =============================

class HessianMatrix : public MatrixContainer { 
  public: 
    HessianMatrix();
    //~HessianMatrix();
    bool ParseIndices(char *name);
    bool SumSubManyBodyTerms(class MatrixContainer &MatrixValues, class KeySetsContainer &KeySets, int Order);
    bool SumSubHessians(class HessianMatrix &Fragments, class KeySetsContainer &KeySets, int Order, double sign);
    bool ParseFragmentMatrix(const char *name, const char *prefix, std::string suffix, int skiplines, int skipcolumns);
  private:
    bool IsSymmetric;
};


// ======================================= END =============================================

#endif /*PARSER_HPP_*/
