/*
 * AtomSet.hpp
 *
 *  Created on: Jul 30, 2010
 *      Author: crueger
 */

#ifndef ATOMSET_HPP_
#define ATOMSET_HPP_


#include <functional>
#include <numeric>
#include <algorithm>
#include <boost/foreach.hpp>
#include <limits>

/**
 * A simple mixin to give any STL conforming structure fast Vector abilities
 *
 * TODO: make this work for maps
 */

#include "atom.hpp"

// this tests, whether we actually have a Vector
template <class V>
struct is_atom{};

template <>
struct is_atom<atom*>{
  typedef void wrong_type;
};

template <class Set>
class AtomSetMixin : public Set
{
  // when our set carries something besides a atom* this will produce an error
  typedef typename is_atom<typename Set::value_type>::wrong_type check_for_atom;
public:
  // typedefs for STL conforming structure
  typedef typename Set::iterator iterator;
  typedef typename Set::const_iterator const_iterator;

  AtomSetMixin() :
    Set()
  {}

  AtomSetMixin(const Set& src) :
    Set(src)
  {}
  virtual ~AtomSetMixin(){}

  /**
   * translate all Atoms within this set by a specified amount
   */
  void translate(const Vector &translater);
  void addVelocityAtStep(const Vector velocity, unsigned int step);

  template<class Function>
  void transformNodes(Function f);
  double totalMass() const;
  double totalTemperatureAtStep(unsigned int step) const;
  Vector totalMomentumAtStep(unsigned int step) const;

private:
  template<class Function>
  struct workOnNodePointer {
    workOnNodePointer(Function &_f) : f(_f){}
    void operator()(atom *atom){
      atom->setPosition(f(atom->getPosition()));
    }
    Function &f;
  };

  template<class T>
  struct valueSum {
    valueSum(T (atom::*_f)() const,T startValue) :
      f(_f),
      value(startValue)
    {}
    T operator+(atom *atom){
      return value + (atom->*f)();
    }
    T operator=(T _value){
      value = _value;
      return value;
    }
    T (atom::*f)() const;
    T value;
  };

  template<class T>
  struct stepValueSum {
    stepValueSum(unsigned int _step, T (atom::*_f)(unsigned int) const,T startValue) :
      step(_step),
      f(_f),
      value(startValue)
    {}
    T operator+(atom *atom){
      return value + (atom->*f)(step);
    }
    T operator=(T _value){
      value = _value;
      return value;
    }
    unsigned int step;
    T (atom::*f)(unsigned int) const;
    T value;
  };
};

template<class Set>
inline void AtomSetMixin<Set>::translate(const Vector &translater){
  BOOST_FOREACH(atom *atom,*this){
    *(atom) += translater;
  }
}

template<class Set>
inline void AtomSetMixin<Set>::addVelocityAtStep(const Vector velocity, unsigned int step){
  BOOST_FOREACH(atom *atom,*this){
    atom->Trajectory.U.at(step) += velocity;
  }
}

template<class Set>
template<class Function>
inline void AtomSetMixin<Set>::transformNodes(Function f){
  std::for_each(this->begin(),
                this->end(),
                AtomSetMixin::workOnNodePointer<Function>(f));
}

template<class Set>
inline double AtomSetMixin<Set>::totalMass() const{
  return accumulate(this->begin(),this->end(),valueSum<double>(&atom::getMass,0)).value;
}

template<class Set>
inline double AtomSetMixin<Set>::totalTemperatureAtStep(unsigned int step) const{
  return accumulate(this->begin(),this->end(),stepValueSum<double>(step,&atom::getKineticEnergy,0)).value;
}

template<class Set>
inline Vector AtomSetMixin<Set>::totalMomentumAtStep(unsigned int step) const{
  return accumulate(this->begin(),this->end(),stepValueSum<Vector>(step,&atom::getMomentum,Vector())).value;
}

// allows simpler definition of AtomSets
#define ATOMSET(container_type) AtomSetMixin<container_type<atom*> >

#endif /* ATOMSET_HPP_ */
