/*
 * AnalysisCorrelationToSurfaceUnitTest.cpp
 *
 *  Created on: Oct 13, 2009
 *      Author: heber
 */

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "analysis_correlation.hpp"
#include "AnalysisCorrelationToSurfaceUnitTest.hpp"

#include "atom.hpp"
#include "boundary.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "linkedcell.hpp"
#include "periodentafel.hpp"
#include "tesselation.hpp"

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AnalysisCorrelationToSurfaceUnitTest );

void AnalysisCorrelationToSurfaceUnitTest::setUp()
{
  atom *Walker = NULL;

  // init private all pointers to zero
  TestMolecule = NULL;
  hydrogen = NULL;
  tafel = NULL;
  surfacemap = NULL;
  binmap = NULL;
  Surface = NULL;
  LC = NULL;


  // construct element
  hydrogen = new element;
  hydrogen->Z = 1;
  strcpy(hydrogen->name, "hydrogen");
  hydrogen->symbol[0] = 'H';

  // construct periodentafel
  tafel = new periodentafel;
  tafel->AddElement(hydrogen);

  // construct molecule (tetraeder of hydrogens)
  TestMolecule = new molecule(tafel);
  Walker = new atom();
  Walker->type = hydrogen;
  Walker->node->Init(1., 0., 1. );
  TestMolecule->AddAtom(Walker);
  Walker = new atom();
  Walker->type = hydrogen;
  Walker->node->Init(0., 1., 1. );
  TestMolecule->AddAtom(Walker);
  Walker = new atom();
  Walker->type = hydrogen;
  Walker->node->Init(1., 1., 0. );
  TestMolecule->AddAtom(Walker);
  Walker = new atom();
  Walker->type = hydrogen;
  Walker->node->Init(0., 0., 0. );
  TestMolecule->AddAtom(Walker);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule->AtomCount, 4 );

  // init tesselation and linked cell
  Surface = new Tesselation;
  TestMolecule->TesselStruct = Surface;
  FindNonConvexBorder((ofstream *)&cout, TestMolecule, LC, 2.5, NULL);
  LC = new LinkedCell(TestMolecule, 5.);
  CPPUNIT_ASSERT_EQUAL( (size_t)4, Surface->PointsOnBoundary.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)6, Surface->LinesOnBoundary.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, Surface->TrianglesOnBoundary.size() );

  // init maps
  surfacemap = CorrelationToSurface( (ofstream *)&cout, TestMolecule, hydrogen, Surface, LC );
  binmap = NULL;

};


void AnalysisCorrelationToSurfaceUnitTest::tearDown()
{
  if (surfacemap != NULL)
    delete(surfacemap);
  if (binmap != NULL)
    delete(binmap);

  // remove
  delete(TestMolecule);
  // note that Surface and all the atoms are cleaned by TestMolecule
  delete(LC);
  delete(tafel);
  // note that element is cleaned by periodentafel
};


void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceTest()
{
  // do the pair correlation
  CPPUNIT_ASSERT( surfacemap != NULL );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, surfacemap->size() );
};

void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceBinNoRangeTest()
{
  BinPairMap::iterator tester;
  // put pair correlation into bins and check with no range
  binmap = BinData( (ofstream *)&cout, surfacemap, 0.5, 0., 0. );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, binmap->size() );
  //OutputCorrelation ( (ofstream *)&cout, binmap );
  tester = binmap->begin();
  CPPUNIT_ASSERT_EQUAL( 0., tester->first );
  CPPUNIT_ASSERT_EQUAL( 4, tester->second );

};

void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceBinRangeTest()
{
  BinPairMap::iterator tester;
  // ... and check with [0., 2.] range
  binmap = BinData( (ofstream *)&cout, surfacemap, 0.5, 0., 2. );
  CPPUNIT_ASSERT_EQUAL( (size_t)5, binmap->size() );
  //OutputCorrelation ( (ofstream *)&cout, binmap );
  tester = binmap->begin();
  CPPUNIT_ASSERT_EQUAL( 0., tester->first );
  CPPUNIT_ASSERT_EQUAL( 4, tester->second );
  tester = binmap->find(1.);
  CPPUNIT_ASSERT_EQUAL( 1., tester->first );
  CPPUNIT_ASSERT_EQUAL( 0, tester->second );

};

/********************************************** Main routine **************************************/

int main(int argc, char **argv)
{
  // Get the top level suite from the registry
  CppUnit::Test *suite = CppUnit::TestFactoryRegistry::getRegistry().makeTest();

  // Adds the test to the list of test to run
  CppUnit::TextUi::TestRunner runner;
  runner.addTest( suite );

  // Change the default outputter to a compiler error format outputter
  runner.setOutputter( new CppUnit::CompilerOutputter( &runner.result(),
                                                       std::cerr ) );
  // Run the tests.
  bool wasSucessful = runner.run();

  // Return error code 1 if the one of test failed.
  return wasSucessful ? 0 : 1;
};
