/*
 * Eigenspace.hpp
 *
 *  Created on: Nov 22, 2010
 *      Author: heber
 */

#ifndef EIGENSPACE_HPP_
#define EIGENSPACE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <boost/shared_ptr.hpp>

#include <iosfwd>
#include <set>
#include <vector>
#include "MatrixContent.hpp"
#include "VectorContent.hpp"

/** Container for a matrix and its eigenvectors and -values.
 *
 *  Note that a set of eigenvectors always forms a subspace within a vector
 *  space, the so-called eigenspace. Hence, the name of this class is
 *  eigenspace.
 *
 *  Eigenvector are stored internally as a matrix. A returned set of
 *  eigenvectors is just a set of VectorViewContent on row vectors in
 *  this matrix.
 */
class Eigenspace
{
  friend std::ostream & operator<<(std::ostream &ost, const Eigenspace &_s);
public:
  typedef std::vector< boost::shared_ptr<VectorContent> > eigenvectorset;
  typedef std::vector< double > eigenvalueset;
  typedef std::set<size_t> indexset;

  Eigenspace(const indexset &_Indices);
  Eigenspace(const indexset &_Indices, const MatrixContent &_matrix);
  virtual ~Eigenspace();

  void calculateEigenSubspace();

  // accessors
  const MatrixContent & getEigenspaceMatrix() const;
  const VectorContent & getEigenvector(const size_t i) const;
  const MatrixContent & getEigenvectorMatrix() const;
  const double getEigenvalue(const size_t i) const;
  const eigenvalueset & getEigenvalues() const;
  const indexset & getIndices() const;
  void setEigenspaceMatrix(const MatrixContent &_content);

  void setEigenpairs(const eigenvectorset &CurrentEigenvectors, const eigenvalueset &CurrentEigenvalues);

  bool contains(const Eigenspace &_indexset);

protected:
  const indexset Indices;
  eigenvectorset Eigenvectors;
  eigenvalueset Eigenvalues;
  MatrixContent EigenspaceMatrix;
  MatrixContent EigenvectorMatrix;

private:
  void createEigenvectorViews();

};

std::ostream & operator<<(std::ostream &ost, const Eigenspace &_s);

#endif /* EIGENSPACE_HPP_ */
