/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * LinkedCell_View.cpp
 *
 *  Created on: Nov 15, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "LinkedCell_View.hpp"

#include "CodePatterns/Log.hpp"

#include "Atom/TesselPoint.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "LinkedCell.hpp"
#include "LinkedCell_Model.hpp"
#include "tripleIndex.hpp"

using namespace LinkedCell;

LinkedCell_View::LinkedCell_View(const LinkedCell_Model &_LC) :
    LC(_LC)
{}

LinkedCell_View::~LinkedCell_View()
{
}


/** Return at least as many points as are inside a sphere of \a radius around \a center.
 *
 * \sa LinkedCell_View::getPointsInsideSphere()
 *
 * @param radius radius of sphere
 * @param center center of sphere
 * @return a list containing at least all points inside described sphere
 */
LinkedList LinkedCell_View::getAllNeighbors(const double radius, const Vector &center) const
{
  LinkedList TesselList;  // we do not need a set, as nodes are uniquely associated to a cell.

  // get relative bounds
  const tripleIndex step = LC.getStep(radius);
  const tripleIndex index = LC.getIndexToVector(center);
  LinkedCell_Model::LinkedCellNeighborhoodBounds neighbors =
      LC.getNeighborhoodBounds(index, step);

  tripleIndex n;
  for (n[0] = 0; n[0] < neighbors.second[0]; n[0]++)
    for (n[1] = 0; n[1] < neighbors.second[1]; n[1]++)
      for (n[2] = 0; n[2] < neighbors.second[2]; n[2]++) {
        tripleIndex absolute_n = neighbors.first + n;
        if (!LC.checkArrayBounds(absolute_n))
          LC.applyBoundaryConditions(absolute_n);
        const LinkedCell &List = LC.getCell(absolute_n);
        LOG(1, "Current cell is " << neighbors.first << " plus " << n << ", yielding " << absolute_n << ".");
        for (LinkedCell::const_iterator Runner = List.begin(); Runner != List.end(); Runner++)
          TesselList.push_back(*Runner);
      }
  return TesselList;
}

LinkedList LinkedCell_View::getPointsInsideSphere(const double radius, const Vector &center) const
{
  // get overly much points
  const LinkedList TesselList = getAllNeighbors(radius, center);
  LinkedList ReturnList;

  // remove all unnecessary ones
  for (LinkedList::const_iterator iter = TesselList.begin(); iter != TesselList.end(); ++iter) {
    if (center.DistanceSquared((*iter)->getPosition()) <= radius*radius)
      ReturnList.push_back(*iter);
  }

  return ReturnList;
}
