/** \file FormatParserStorage.cpp
 *
 *  date: Jun, 22 2010
 *  author: heber
 *
 */

#include <iostream>
#include <fstream>

#include "Parser/FormatParserStorage.hpp"

#include "Parser/FormatParser.hpp"
#include "Parser/MpqcParser.hpp"
#include "Parser/PcpParser.hpp"
#include "Parser/TremoloParser.hpp"
#include "Parser/XyzParser.hpp"

#include "log.hpp"
#include "verbose.hpp"

#include "Helpers/Assert.hpp"

#include "Patterns/Singleton_impl.hpp"

/** Constructor of class FormatParserStorage.
 */
FormatParserStorage::FormatParserStorage()
{
  ParserList.resize(ParserTypes_end, NULL);
  ParserPresent.resize(ParserTypes_end, false);
  ParserSuffix.resize(ParserTypes_end, "");

  ParserSuffix[mpqc] = "conf.in";
  ParserSuffix[pcp] = "conf";
  ParserSuffix[tremolo] = "conf.data";
  ParserSuffix[xyz] = "conf.xyz";
}

/** Destructor of class FormatParserStorage.
 * Free all stored FormatParsers. 
 * Save on Exit.
 */
FormatParserStorage::~FormatParserStorage()
{
  for (ParserTypes iter = ParserTypes_begin; iter < ParserTypes_end; ++iter)
    if (ParserPresent[iter])
      delete ParserList[iter];
}

/** Sets the filename of all current parsers in storage to prefix.suffix.
 * \param &prefix prefix to use.
 */
void FormatParserStorage::SetOutputPrefixForAll(std::string &_prefix)
{
  prefix=_prefix;
};


void FormatParserStorage::SaveAll()
{
  std::string filename;
  for (ParserTypes iter = ParserTypes_begin; iter < ParserTypes_end; ++iter)
    if (ParserPresent[iter]) {
      filename = prefix;
      filename += ".";
      filename += ParserSuffix[iter];
      std::ofstream *file = new std::ofstream(filename.c_str());
      ParserList[iter]->setOstream((std::ostream *)file);
    }
}


/** Adds an MpqcParser to the storage.
 */
void FormatParserStorage::AddMpqc()
{
  if (!ParserPresent[mpqc])
    ParserList[mpqc] = dynamic_cast<FormatParser *>(new MpqcParser);
  else
    DoeLog(1) && (eLog() << Verbose(1) << "Parser mpqc is already present." << endl);
}

/** Adds an PcpParser to the storage.
 */
void FormatParserStorage::AddPcp()
{
  if (!ParserPresent[pcp])
    ParserList[pcp] = new PcpParser();
  else
    DoeLog(1) && (eLog() << Verbose(1) << "Parser pcp is already present." << endl);
}


/** Adds an TremoloParser to the storage.
 */
void FormatParserStorage::AddTremolo()
{
  if (!ParserPresent[tremolo])
    ParserList[tremolo] = new TremoloParser();
  else
    DoeLog(1) && (eLog() << Verbose(1) << "Parser tremolo is already present." << endl);
}


/** Adds an XyzParser to the storage.
 */
void FormatParserStorage::AddXyz()
{
  if (!ParserPresent[xyz])
    ParserList[xyz] = new XyzParser();
  else
    DoeLog(1) && (eLog() << Verbose(1) << "Parser xyz is already present." << endl);
}


CONSTRUCT_SINGLETON(FormatParserStorage)
