/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ForceAnnealingAction.cpp
 *
 *  Created on: Aug 02, 2014
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "Actions/ActionExceptions.hpp"
#include "Actions/MakroAction.hpp"
#include "Actions/UndoRedoHelpers.hpp"
#include "Atom/atom.hpp"
#include "Atom/AtomicInfo.hpp"
#include "Atom/AtomSet.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Dynamics/ForceAnnealing.hpp"
#include "molecule.hpp"
#include "World.hpp"
#include "WorldTime.hpp"

#include <vector>
#include <iostream>
#include <fstream>
#include <string>

#include "Actions/MoleculeAction/ForceAnnealingAction.hpp"

using namespace MoleCuilder;

enum VectorIndexType {
  PositionIndex=0,
  VelocityIndex=1,
  ForceIndex=2
};

// and construct the stuff
#include "ForceAnnealingAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr MoleculeForceAnnealingAction::performCall() {
  AtomSetMixin<std::vector<atom *> > set(World::getInstance().getSelectedAtoms());
  if (set.empty()) {
    STATUS("No atoms selected.");
    return Action::failure;
  }

  size_t CurrentStep = WorldTime::getInstance().getTime();
  if (CurrentStep == 0) {
    ELOG(1, "WorldTime must be at least at step 1 already, use step-world-time if necessary.");
    return Action::failure;
  }

  // first, we need to sort the mixin according to their ids (as selected atoms are sorted
  // according to their arbitrary address in memory)
  set.sortByIds();

  // create undo state for all selected atoms (undo info)
  std::vector< std::vector<AtomicInfo> > UndoInfo(2);
  for (int i=0;i<2;++i) {
    UndoInfo[i].reserve(set.size());
    {
      for (World::AtomSelectionConstIterator iter = World::getInstance().beginAtomSelection();
          iter != World::getInstance().endAtomSelection();
          ++iter)
          UndoInfo[i].push_back(AtomicInfo(*(iter->second), CurrentStep-i));
    }
  }
  std::vector<int> UndoTrajectorySize;
  UndoTrajectorySize.reserve(set.size());
  for (World::AtomSelectionConstIterator iter = World::getInstance().beginAtomSelection();
      iter != World::getInstance().endAtomSelection();
      ++iter)
    UndoTrajectorySize.push_back(iter->second->getTrajectorySize());

  // instantiate optimizer
  ForceAnnealing<std::vector<atom *> > optimizer(
      set,
      params.deltat.get(),
      true,
      params.steps.get(),
      params.MaxDistance.get(),
      params.DampingFactor.get());

  // parse forces into last step (assuming we stepped on already)
  if (!params.forcesfile.get().string().empty()) {
    LOG(1, "Parsing forces file.");
    if (!optimizer.parseForcesFile(params.forcesfile.get().string().c_str(), CurrentStep-1))
      LOG(2, "File " << params.forcesfile.get() << " not found.");
    else
      LOG(2, "File " << params.forcesfile.get() << " found and parsed.");
  }

  // perform optimization step
  LOG(1, "Structural optimization.");
  const bool StopStatus = optimizer(CurrentStep, 1, params.UseBondGraph.get());
  STATUS("Successfully optimized structure by one step.");

  if (StopStatus && ActionQueue::getInstance().isMakroAction()) {
    // send stop signal if we are taking part in MakroAction
    MakroAction * const makroaction =
        dynamic_cast<MakroAction *>(
            const_cast<Action *>(
                &ActionQueue::getInstance().getCurrentAction()));
    if (makroaction != NULL) {
      makroaction->setLoop(makroaction->getStep());
    } else {
      ELOG(2, "ActionQueue said we are inside process, but current Action is not a process?");
      // do nothing
    }
  }

  std::vector< std::vector<AtomicInfo> > RedoInfo(2);
  for (int i=0;i<2;++i) {
    RedoInfo[i].reserve(set.size());
    {
      for (World::AtomSelectionConstIterator iter = World::getInstance().beginAtomSelection();
          iter != World::getInstance().endAtomSelection();
          ++iter)
        RedoInfo[i].push_back(AtomicInfo(*(iter->second), CurrentStep-i));
    }
  }

  MoleculeForceAnnealingState *UndoState =
    new MoleculeForceAnnealingState(UndoInfo, UndoTrajectorySize, RedoInfo, params);

  return ActionState::ptr(UndoState);
}

ActionState::ptr MoleculeForceAnnealingAction::performUndo(ActionState::ptr _state) {
  MoleculeForceAnnealingState *state =
      assert_cast<MoleculeForceAnnealingState*>(_state.get());
  const size_t CurrentStep = WorldTime::getInstance().getTime();

  // set stored old state and remove current one from trajectory if we set it
  for (int i=0;i<2;++i) {
    for(size_t j=0;j<state->UndoInfo[i].size();++j) {
      const AtomicInfo &_atominfo = state->UndoInfo[i][j];
      const atomId_t id = _atominfo.getId();
      atom * const _atom = World::getInstance().getAtom(AtomById(id));
      ASSERT( _atom != NULL,
          "MoleCuilder::SetAtomsFromAtomicInfo() - cannot find atom with id "
          +toString(id)+" in the world.");
      if (state->UndoTrajectorySize[j] > CurrentStep-i)
        _atominfo.setAtom( *_atom, CurrentStep-i );
      else
        _atom->removeSteps(state->UndoTrajectorySize[j], CurrentStep-i);
    }
  }

  return ActionState::ptr(_state);
}

ActionState::ptr MoleculeForceAnnealingAction::performRedo(ActionState::ptr _state){
  MoleculeForceAnnealingState *state =
      assert_cast<MoleculeForceAnnealingState*>(_state.get());
  const size_t CurrentStep = WorldTime::getInstance().getTime();

  // set stored new state
  for (int i=0;i<2;++i)
    SetAtomsFromAtomicInfo(state->RedoInfo[i], CurrentStep-i);

  return ActionState::ptr(_state);
}

bool MoleculeForceAnnealingAction::canUndo() {
  return true;
}

bool MoleculeForceAnnealingAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
