/*
 * atom_atominfo.hpp
 *
 *  Created on: Oct 19, 2009
 *      Author: heber
 */

#ifndef ATOM_ATOMINFO_HPP_
#define ATOM_ATOMINFO_HPP_


using namespace std;

/*********************************************** includes ***********************************/

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <vector>

#include "LinearAlgebra/Vector.hpp"
#include "LinearAlgebra/VectorInterface.hpp"

/****************************************** forward declarations *****************************/

class AtomInfo;
class config;
class element;
class ForceMatrix;
class RealSpaceMatrix;

/********************************************** declarations *******************************/

class AtomInfo : public VectorInterface {

public:
  AtomInfo();
  AtomInfo(const AtomInfo &_atom);
  AtomInfo(const VectorInterface &_v);
  virtual ~AtomInfo();

  /** Getter for AtomicElement.
   *
   * @return constant reference to AtomicElement
   */
  const element *getType() const;
  /** Setter for AtomicElement.
   *
   * @param _type new element by pointer to set
   */
  void setType(const element *_type);
  /** Setter for AtomicElement.
   *
   * @param _typenr new element by index to set
   */
  void setType(const int _typenr);

  /** Getter for AtomicVelocity.
   *
   * Current time step is used.
   *
   * @return constant reference to AtomicVelocity
   */
//  Vector& getAtomicVelocity();
  /** Getter for AtomicVelocity.
   *
   * @param _step time step to return
   * @return constant reference to AtomicVelocity
   */
//  Vector& getAtomicVelocity(const int _step);
  /** Getter for AtomicVelocity.
   *
   * Current time step is used.
   *
   * @return constant reference to AtomicVelocity
   */
  const Vector& getAtomicVelocity() const;
  /** Getter for AtomicVelocity.
   *
   * @param _step time step to return
   * @return constant reference to AtomicVelocity
   */
  const Vector& getAtomicVelocityAtStep(const unsigned int _step) const;
  /** Setter for AtomicVelocity.
   *
   * Current time step is used.
   *
   * @param _newvelocity new velocity to set
   */
  void setAtomicVelocity(const Vector &_newvelocity);
  /** Setter for AtomicVelocity.
   *
   * @param _step time step to set
   * @param _newvelocity new velocity to set
   */
  void setAtomicVelocityAtStep(const unsigned int _step, const Vector &_newvelocity);

  /** Getter for AtomicForce.
   *
   * Current time step is used.
   *
   * @return constant reference to AtomicForce
   */
  const Vector& getAtomicForce() const;
  /** Getter for AtomicForce.
   *
   * @param _step time step to return
   * @return constant reference to AtomicForce
   */
  const Vector& getAtomicForceAtStep(const unsigned int _step) const;
  /** Setter for AtomicForce.
   *
   * Current time step is used.
   *
   * @param _newvelocity new force vector to set
   */
  void setAtomicForce(const Vector &_newforce);
  /** Setter for AtomicForce.
   *
   * @param _step time step to set
   * @param _newvelocity new force vector to set
   */
  void setAtomicForceAtStep(const unsigned int _step, const Vector &_newforce);

  /** Getter for FixedIon.
   *
   * @return constant reference to FixedIon
   */
  bool getFixedIon() const;
  /** Setter for FixedIon.
   *
   * @param _fixedion new state of FixedIon
   */
  void setFixedIon(const bool _fixedion);

  ///// manipulation of the atomic position

  // Accessors ussually come in pairs... and sometimes even more than that
  /** Getter for AtomicPosition.
   *
   * Current time step is used.
   *
   * @param i component of vector
   * @return i-th component of atomic position
   */
  const double& operator[](size_t i) const;
  /** Getter for AtomicPosition.
   *
   * Current time step is used.
   *
   * \sa operator[], this is if instance is a reference.
   *
   * @param i component of vector
   * @return i-th component of atomic position
   */
  const double& at(size_t i) const;
  /** Getter for AtomicPosition.
   *
   * \sa operator[], this is if instance is a reference.
   *
   * @param i index of component of AtomicPosition
   * @param _step time step to return
   * @return atomic position at time step _step
   */
  const double& atStep(size_t i, unsigned int _step) const;
  /** Setter for AtomicPosition.
   *
   * Current time step is used.
   *
   * @param i component to set
   * @param value value to set to
   */
  void set(size_t i, const double value);
  /** Setter for AtomicPosition.
   *
   * @param i component to set
   * @param _step time step to set
   * @param value value to set to
   */
  void setAtStep(size_t i, unsigned int _step, const double value);
  /** Getter for AtomicPosition.
   *
   * Current time step is used.
   *
   * @return atomic position
   */
  const Vector& getPosition() const;
  /** Getter for AtomicPosition.
   *
   * @param _step time step to return
   * @return atomic position at time step _step
   */
  const Vector& getPositionAtStep(unsigned int _step) const;

  // Assignment operator
  /** Setter for AtomicPosition.
   *
   * Current time step is used.
   *
   * @param _vector new position to set
   */
  void setPosition(const Vector& _vector);
  /** Setter for AtomicPosition.
   *
   * @param _step time step to set
   * @param _vector new position to set for time step _step
   */
  void setPositionAtStep(const unsigned int _step, const Vector& _vector);
  class VectorInterface &operator=(const Vector& _vector);

  // operators for mathematical operations
  const VectorInterface& operator+=(const Vector& b);
  const VectorInterface& operator-=(const Vector& b);
  Vector const operator+(const Vector& b) const;
  Vector const operator-(const Vector& b) const;

  void Zero();
  void One(const double one);
  void LinearCombinationOfVectors(const Vector &x1, const Vector &x2, const Vector &x3, const double * const factors);

  double distance(const Vector &point) const;
  double DistanceSquared(const Vector &y) const;
  double distance(const VectorInterface &_atom) const;
  double DistanceSquared(const VectorInterface &_atom) const;

  void ScaleAll(const double *factor);
  void ScaleAll(const Vector &factor);
  void Scale(const double factor);

  // operations for trajectories
  void ResizeTrajectory(size_t MaxSteps);
  size_t getTrajectorySize() const;
  void CopyStepOnStep(const unsigned int dest, const unsigned int src);
  void VelocityVerletUpdate(int nr, const unsigned int MDSteps, config *configuration, ForceMatrix *Force, const size_t offset);
  double getKineticEnergy(const unsigned int step) const;
  Vector getMomentum(const unsigned int step) const;
  double getMass() const;

  std::ostream & operator << (std::ostream &ost) const;

protected:
  // make these protected only such that deriving atom class still has full
  // access needed for clone and alike
  std::vector<Vector> AtomicPosition;       //!< coordinate vector of atom, giving last position within cell
  std::vector<Vector> AtomicVelocity;       //!< velocity vector of atom, giving last velocity within cell
  std::vector<Vector> AtomicForce;       //!< Force vector of atom, giving last force within cell

private:
  const element *AtomicElement;  //!< pointing to element
  bool FixedIon;
};

std::ostream & operator << (std::ostream &ost, const AtomInfo &a);

const AtomInfo& operator*=(AtomInfo& a, const double m);
AtomInfo const operator*(const AtomInfo& a, const double m);
AtomInfo const operator*(const double m, const AtomInfo& a);

#endif /* ATOM_ATOMINFO_HPP_ */
