/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * CorrectBondDegreeAction.cpp
 *
 *  Created on: Jul 31, 2014
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "Descriptors/AtomSelectionDescriptor.hpp"
#include "Graph/BondGraph.hpp"
#include "molecule.hpp"
#include "World.hpp"

#include <iostream>
#include <list>
#include <string>

#include "Actions/GraphAction/CorrectBondDegreeAction.hpp"

using namespace MoleCuilder;

// Storing undo state
struct BondInfo_t {
  atomId_t leftatom;
  atomId_t rightatom;
  size_t degree;
};

// and construct the stuff
#include "CorrectBondDegreeAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr GraphCorrectBondDegreeAction::performCall() {
  BondGraph *BG = World::getInstance().getBondGraph();
  ASSERT(BG != NULL, "GraphCorrectBondDegreeAction: BondGraph is NULL.");

  // count all present bonds
  World::AtomComposite Set = World::getInstance().getAllAtoms(AtomsBySelection());
  std::vector<BondInfo_t> bonds;
  {
    size_t count_bonds = 0;
    for (World::AtomComposite::const_iterator iter = Set.begin();
        iter != Set.end();
        ++iter) {
      const atom * const Walker = *iter;
      count_bonds += Walker->getListOfBonds().size();
    }
    bonds.reserve(count_bonds/2);
  }
  // Storing undo info
  for (World::AtomComposite::const_iterator iter = Set.begin();
      iter != Set.end();
      ++iter) {
    const atom * const Walker = *iter;
    const BondList& ListOfBonds = Walker->getListOfBonds();
    for (BondList::const_iterator bonditer = ListOfBonds.begin();
        bonditer != ListOfBonds.end();
        ++bonditer) {
      const bond::ptr &CurrentBond = *bonditer;
      // if both atoms are in selected set, we check ids otherwise not
      if (((!World::getInstance().isSelected(CurrentBond->leftatom))
          || (!World::getInstance().isSelected(CurrentBond->rightatom)))
          || (CurrentBond->leftatom->getId() < CurrentBond->rightatom->getId())) {
        BondInfo_t BondInfo;
        BondInfo.leftatom = CurrentBond->leftatom->getId();
        BondInfo.rightatom = CurrentBond->rightatom->getId();
        BondInfo.degree = CurrentBond->getDegree();
        bonds.push_back(BondInfo);
      }
    }
  }
  GraphCorrectBondDegreeState *UndoState = new GraphCorrectBondDegreeState(bonds, params);

  // now recreate adjacency
  BG->CorrectBondDegree(Set);

  return ActionState::ptr(UndoState);
}

ActionState::ptr GraphCorrectBondDegreeAction::performUndo(ActionState::ptr _state) {
  GraphCorrectBondDegreeState *state = assert_cast<GraphCorrectBondDegreeState*>(_state.get());

  BondGraph *BG = World::getInstance().getBondGraph();
  ASSERT(BG != NULL, "GraphCorrectBondDegreeAction: BondGraph is NULL.");

  // remove all bonds in the set
  World::AtomComposite Set = World::getInstance().getAllAtoms(AtomsBySelection());
  BG->cleanAdjacencyList(Set);

  // recreate from stored info
  const size_t CurrentTime = WorldTime::getTime();
  std::vector<BondInfo_t> &bonds = state->bonds;
  for(std::vector<BondInfo_t>::const_iterator iter = bonds.begin();
      iter != bonds.end(); ++iter) {
    atom * const Walker = World::getInstance().getAtom(AtomById(iter->leftatom));
    ASSERT( Walker != NULL,
        "GraphCorrectBondDegreeAction::performUndo() - "+toString(iter->leftatom)+" missing.");
    atom * const OtherWalker = World::getInstance().getAtom(AtomById(iter->rightatom));
    ASSERT( OtherWalker != NULL,
        "GraphCorrectBondDegreeAction::performUndo() - "+toString(iter->rightatom)+" missing.");
    bond::ptr CurrentBond = Walker->addBond(CurrentTime, OtherWalker);
    CurrentBond->setDegree(iter->degree);
  }
  return ActionState::ptr(_state);
}

ActionState::ptr GraphCorrectBondDegreeAction::performRedo(ActionState::ptr _state){
  BondGraph *BG = World::getInstance().getBondGraph();
  ASSERT(BG != NULL, "GraphCorrectBondDegreeAction: BondGraph is NULL.");

  // now recreate adjacency
  World::AtomComposite Set = World::getInstance().getAllAtoms(AtomsBySelection());
  BG->CorrectBondDegree(Set);

  return ActionState::ptr(_state);
}

bool GraphCorrectBondDegreeAction::canUndo() {
  return true;
}

bool GraphCorrectBondDegreeAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
