/*
 * AllLevelOrthogonalSummator.hpp
 *
 *  Created on: 29.07.2012
 *      Author: heber
 */

#ifndef ALLLEVELORTHOGONALSUMMATOR_HPP_
#define ALLLEVELORTHOGONALSUMMATOR_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <vector>

#include "CodePatterns/Assert.hpp"

#include "Fragmentation/Summation/IndexSetContainer.hpp"
#include "Fragmentation/Summation/SubsetMap.hpp"
#include "Fragmentation/Summation/OrthogonalSummator.hpp"

#include "Fragmentation/Summation/printKeyNames.hpp"

/** Tiny template functor to use OrthogonalSummation, sum up each level, and store the
 * result in a given vector.
 *
 */
template <typename MapType>
struct AllLevelOrthogonalSummator {
  /** Constructor takes the arguments that \a OrthogonalSummator also needs and stores
   * them internally.
   *
   * \param _subsetmap map with hierarchy of IndexSet's
   * \param _data MPQCData converted to MPQCDataMap_t type, associated to the jobid
   * \param _container container of IndexSet's such that each set has correct order
   *        to job id and hence to _data.
   * \param _MatrixNrLookup lookup from job id to ordering in above vectors
   * \param _results vector place results into
   */
  AllLevelOrthogonalSummator(
      SubsetMap::ptr &_subsetmap,
      const std::map<JobId_t, MapType> &_data,
      const IndexSetContainer::Container_t &_container,
      const std::map< JobId_t, size_t > &_MatrixNrLookup,
      std::vector<MapType> &_results) :
    subsetmap(_subsetmap),
    data(_data),
    container(_container),
    MatrixNrLookup(_MatrixNrLookup),
    results(_results)
  {
    ASSERT( results.size() >= subsetmap->getMaximumSubsetLevel(),
        "AllLevelOrthogonalSummator() - result vector is not large enough.");
  }

  /** Operator that calls on OrthogonalSummator and prints the value.
   *
   * \note the parameter is needed for boost::mpl::for_each but is not
   * used here.
   */
  template <typename MapKey>
  void operator()(MapKey &) {
    // We retrieve the type of the MPQCData member variable from the boost::fusion::map.
    typedef typename boost::fusion::result_of::value_at_key<MapType, MapKey>::type MapValue;
    // create OrthogonalSummator instance
    OrthogonalSummator<MapType, MapKey> sum_value(
        subsetmap, data, container, MatrixNrLookup
        );
    const size_t MaxLevel = subsetmap->getMaximumSubsetLevel();
    for (size_t level=1; level <= MaxLevel; ++level) {
      MapType &LevelResults = results[level-1];
      // sum up and store in results
      const MapValue value = sum_value(level);
      boost::fusion::at_key<MapKey>(LevelResults) = value;
      // print value
      //LOG(0, "STATUS: Level " << level << " resulting " << printKeyNames::printName<MapKey>() << " is " << value << ".");
    }
  }

private:
  //!> Hierarchy of IndexSet's
  SubsetMap::ptr &subsetmap;
  //!> map from jobids to data converted from MPQCData
  const std::map<JobId_t, MapType> &data;
  //!> container with all IndexSet's
  const IndexSetContainer::Container_t &container;
  //!> lookup map from job ids to ordering in above vectors
  const std::map< JobId_t, size_t > &MatrixNrLookup;
  //!> vector of results
  std::vector<MapType> &results;
};

#endif /* ALLLEVELORTHOGONALSUMMATOR_HPP_ */
