/*
 * MatrixContent.hpp
 *
 *  Created on: Jul 2, 2010
 *      Author: crueger
 */

#ifndef MATRIXCONTENT_HPP_
#define MATRIXCONTENT_HPP_

/** MatrixContent is a wrapper for gsl_matrix.
 *
 * The GNU Scientific Library contaisn some very well written routines for
 * linear algebra problems. However, it's syntax is C and hence it does not
 * lend itself to readable written code, i.e. C = A * B, where A, B, and C
 * are all matrices. Writing code this way is very convenient, concise and
 * also in the same manner as one would type in MatLab.
 * However, with C++ and its feature to overload functions and its operator
 * functions such syntax is easy to create.
 *
 * Hence, this class is a C++ wrapper to gsl_matrix. There already some other
 * libraries, however they fall short for the following reasons:
 * -# gslwrap: not very well written and uses floats instead of doubles
 * -# gslcpp: last change is from 2007 and only very few commits
 * -# o2scl: basically, the same functionality as gsl only written in C++,
 *    however it uses GPLv3 license which is inconvenient for MoleCuilder.
 *
 * <h1>Howto use MatrixContent</h1>
 *
 * One should not use MatrixContent directly but either have it as a member
 * variable in a specialized class or inherit from it.
 *
 */

#include <gsl/gsl_matrix.h>

class Vector;

class MatrixContent
{
  friend Vector operator*(const MatrixContent &mat,const Vector &vec);
  friend class Matrix;
public:
  MatrixContent(size_t rows, size_t columns);
  MatrixContent(size_t rows, size_t columns, const double *src);
  MatrixContent(gsl_matrix *&src);
  MatrixContent(const MatrixContent &src);
  MatrixContent(const MatrixContent *src);
  ~MatrixContent();

  // Accessing
  double &at(size_t i, size_t j);
  const double at(size_t i, size_t j) const;
  void set(size_t i, size_t j, const double value);

  // Transformations
  MatrixContent transpose() const;
  MatrixContent &transpose();
  gsl_vector* transformToEigenbasis();

  // Initializing
  void setIdentity();
  void setZero();

  // Operators
  MatrixContent &operator=(const MatrixContent &src);
  const MatrixContent &operator+=(const MatrixContent &rhs);
  const MatrixContent &operator-=(const MatrixContent &rhs);
  const MatrixContent &operator*=(const MatrixContent &rhs);
  const MatrixContent operator*(const MatrixContent &rhs) const;
  const MatrixContent &operator*=(const double factor);
  bool operator==(const MatrixContent &rhs) const;


private:
  gsl_matrix * content;
  const size_t rows;      // store for internal purposes
  const size_t columns;  // store for internal purposes
};

const MatrixContent operator*(const double factor,const MatrixContent& mat);
const MatrixContent operator*(const MatrixContent &mat,const double factor);
Vector operator*(const MatrixContent &mat,const Vector &vec);

#endif /* MATRIXCONTENT_HPP_ */
