/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtMainWindow.cpp
 *
 *  Created on: Jan 14, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "QtMainWindow.hpp"

#include<Qt/qapplication.h>
#include<Qt/qlabel.h>
#include<Qt/qstring.h>
#include<Qt/qmenubar.h>
#include<Qt/qsplitter.h>
#include<Qt/qtoolbar.h>
#include<Qt/qsettings.h>
#include<Qt/qmessagebox.h>
#include<Qt/qboxlayout.h>
#include<Qt/qevent.h>

#include<iostream>
#include<map>

#include<boost/bind.hpp>

#include "Menu/Qt4/QtMenu.hpp"
#include "Views/Qt4/MoleculeList/QtMoleculeList.hpp"
#include "Views/Qt4/MoleculeList/QtMoleculeListView.hpp"
#include "Views/Qt4/QtElementList.hpp"
#include "Views/Qt4/QtFragmentList.hpp"
#include "Views/Qt4/QtHomologyList.hpp"
#include "UIElements/Qt4/InstanceBoard/QtObservedInstanceBoard.hpp"
#include "Views/Qt4/QtLogBox.hpp"
#include "Views/Qt4/QtShapeController.hpp"
#include "Views/Qt4/QtInfoBox.hpp"
#include "Views/Qt4/QtSelectionChangedAgent.hpp"
#include "Views/Qt4/QtStatusBar.hpp"
#include "Views/Qt4/QtTimeLine.hpp"
#include "Views/Qt4/QtToolBar.hpp"
#include "Views/Qt4/Qt3D/GLWorldView.hpp"

#include "CodePatterns/MemDebug.hpp"
#include "CodePatterns/Verbose.hpp"

#include "Atom/atom.hpp"
#include "molecule.hpp"
#include "Actions/Action.hpp"
#include "Actions/ActionQueue.hpp"
#include "Parser/ChangeTracker.hpp"
#include "Parser/FormatParserStorage.hpp"
#include "Actions/WorldAction/OutputAction.hpp"
#include "Actions/WorldAction/OutputAsAction.hpp"

#include "version.h"

using namespace MoleCuilder;

QtMainWindow::QtMainWindow(QApplication *_theApp) :
    theApp(_theApp)
{
  qRegisterMetaType<atomId_t>("atomId_t");
  qRegisterMetaType<moleculeId_t>("moleculeId_t");

  Q_INIT_RESOURCE(icons);
  setWindowTitle(MOLECUILDERVERSION);
  theApp->setWindowIcon(QIcon(QPixmap(":/molecuildergui_logo.png")));
  QCoreApplication::setOrganizationName("ins");
  QCoreApplication::setOrganizationDomain("ins.uni-bonn.de");
  QCoreApplication::setApplicationName("MoleCuilder");

  InstanceBoard = new QtObservedInstanceBoard(this);

  QSplitter *splitter1 = new QSplitter (Qt::Horizontal, this );
  QSplitter *splitter2 = new QSplitter (Qt::Vertical, splitter1 );
  QSplitter *splitter3 = new QSplitter (Qt::Vertical, splitter1 );
  QTabWidget *worldTab = new QTabWidget(splitter2);
  QWidget *layoutwidget = new QWidget(splitter2);
  QVBoxLayout *layout = new QVBoxLayout(layoutwidget);

  selectionagent = new QtSelectionChangedAgent(this);

  QtMoleculeListView *moleculeListView = new QtMoleculeListView(worldTab);
  moleculeList = new QtMoleculeList(InstanceBoard);
  moleculeListView->setModel(moleculeList);
  moleculeListView->setSelectionChangedAgent(selectionagent);

  elementList = new QtElementList(worldTab);
  homologyList = new QtHomologyList(worldTab);
  fragmentList = new QtFragmentList(worldTab);
  logBox = new QtLogBox(worldTab);
  shapeController = new QtShapeController(worldTab);

  timeline = new QtTimeLine(this);

  infoBox = new QtInfoBox(InstanceBoard);
  glWorldView = new GLWorldView(InstanceBoard, this);
  glWorldView->setSizePolicy( QSizePolicy::Expanding,QSizePolicy::Expanding);
  glWorldView->setMinimumSize( QSize(640,480) );
//  glWorldView->setFocus();
  glWorldView->camera()->setEye( QVector3D(0,3,10));
  glWorldView->setSelectionChangedAgent(selectionagent);

  MainMenu = new QtMenu<QMenuBar>(menuBar(), "");
  MainMenu->init();

  toolbar = new QtToolBar(this);
  glWorldView->addToolBarActions(toolbar);
  toolbar->addFavoriteActionItems(20);
  addToolBar(toolbar);

  setCentralWidget(splitter1);
  splitter1->addWidget(splitter2);
  splitter1->addWidget(splitter3);
  splitter1->setStretchFactor(0, 10);
  splitter3->addWidget(infoBox);
  splitter3->addWidget(shapeController);
  splitter2->addWidget(glWorldView);
  layout->addWidget(timeline);
  layout->addWidget(worldTab);
  splitter2->addWidget(layoutwidget);
  worldTab->addTab(moleculeListView, "Molecules");
  worldTab->addTab(elementList, "All Elements");
  worldTab->addTab(fragmentList, "All Fragments");
  worldTab->addTab(homologyList, "All Homologies");
  worldTab->addTab(logBox, "Log");

  statusBar = new QtStatusBar(this);
  setStatusBar(statusBar);

  QSettings settings;
  settings.beginGroup("MainWindow");
  resize(settings.value("size", QSize(400, 400)).toSize());
  move(settings.value("position", QPoint(200, 200)).toPoint());
  if (settings.value("maximized", false).toBool())
    showMaximized();
  settings.endGroup();

  connect(glWorldView,SIGNAL(hoverChanged(const atomId_t)), infoBox,SLOT(atomHover(const atomId_t)));
  connect(glWorldView,SIGNAL(hoverChanged(const moleculeId_t, int)), infoBox,SLOT(moleculeHover(const moleculeId_t)));
  connect(moleculeList,SIGNAL(moleculesVisibilityChanged(const moleculeId_t,bool)), glWorldView, SIGNAL(moleculesVisibilityChanged(const moleculeId_t,bool)));

  glWorldView->fitCameraToDomain();
}

QtMainWindow::~QtMainWindow()
{
  QSettings settings;
  settings.beginGroup("MainWindow");
  settings.setValue("size", size());
  settings.setValue("position", pos());
  settings.setValue("maximized", isMaximized());
  settings.endGroup();

  menuBar()->clear();
  delete MainMenu;
}

void QtMainWindow::display() {
  this->show();
  this->update();
  theApp->exec();
}

void QtMainWindow::closeEvent(QCloseEvent *event)
{
  if (ChangeTracker::getInstance().hasChanged()){
    int ret = QMessageBox::question(this, tr("MoleCuilder"),
                                    tr("The world has been modified.\n"
                                       "Do you want to save your changes?"),
                                    QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel,
                                    QMessageBox::Save);
    switch(ret){
      case QMessageBox::Cancel:
        event->ignore();
        return;
      case QMessageBox::Save:
        if (FormatParserStorage::getInstance().isAbleToSave())
          MoleCuilder::WorldOutput();
        else{
          try{
            ActionQueue::getInstance().queueAction("output-as");
          }catch(...){
            ELOG(1, "Action failed or cancelled");
          }
        }
        break;
    }
  }
  QMainWindow::closeEvent(event);
}

