/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtMoleculeList.cpp
 *
 *  Created on: Jan 21, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Views/Qt4/QtMoleculeList.hpp"

#include <QMetaMethod>

#include <iostream>

#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "Formula.hpp"
#include "molecule.hpp"
#include "MoleculeListClass.hpp"
#include "Actions/SelectionAction/Molecules/MoleculeByIdAction.hpp"
#include "Actions/SelectionAction/Molecules/NotMoleculeByIdAction.hpp"

using namespace std;

// maybe this should go with the definition of molecules

// some attributes need to be easier to find for molecules
// these attributes are skipped so far
const int QtMoleculeList::COLUMNCOUNT = COLUMNTYPES_MAX;
const char *QtMoleculeList::COLUMNNAMES[QtMoleculeList::COLUMNCOUNT]={"Name","Visibility", "Atoms","Formula","Occurrence"/*,"Size"*/};

QtMoleculeList::QtMoleculeList(QWidget * _parent) :
  QTreeWidget (_parent),
  Observer("QtMoleculeList")
{
  setColumnCount(COLUMNCOUNT);
  setSelectionMode(QAbstractItemView::MultiSelection);

  QStringList header;
  for(int i=0; i<COLUMNCOUNT;++i)
    header << COLUMNNAMES[i];
  setHeaderLabels(header);

  World::getInstance().signOn(this);//, World::MoleculeInserted);
  //World::getInstance().signOn(this, World::MoleculeRemoved);


  dirty = true;
  clearing = false;
  selecting = false;
  refill();

  qRegisterMetaType<QItemSelection>("QItemSelection");
  //connect(this,SIGNAL(cellChanged(int,int)),this,SLOT(moleculeChanged(int,int)));
  connect(selectionModel(),SIGNAL(selectionChanged(QItemSelection, QItemSelection)),this,SLOT(rowsSelected(QItemSelection, QItemSelection)));
  connect(this, SIGNAL(itemChanged(QTreeWidgetItem*, int)), this, SLOT(visibilityChanged(QTreeWidgetItem*, int)));
}

QtMoleculeList::~QtMoleculeList()
{
  World::getInstance().signOff(this);//, World::MoleculeInserted);
  //World::getInstance().signOff(this, World::MoleculeRemoved);
}

void QtMoleculeList::update(Observable *publisher) {

  if (selecting)
    return;

  dirty = true;

  // force an update from Qt...
  clearing = true;
  clear();
  clearing = false;
}

void QtMoleculeList::refill() {
  clearing = true;
  const std::vector<molecule*> &molecules = World::getInstance().getAllMolecules();

  clear();

  // list of (unique) formulas in the world
  std::vector<Formula> formula;

  for (std::vector<molecule*>::const_iterator iter = molecules.begin();
      iter != molecules.end();
      iter++) {

    // find group if already in list
    QTreeWidgetItem *groupItem = NULL;
    for (unsigned int j=0;j<formula.size();j++)
      if ((*iter)->getFormula() == formula[j]){
        groupItem = topLevelItem(j);
        break;
      }

    // new molecule type -> create new group
    if (!groupItem){
      formula.push_back((*iter)->getFormula());
      groupItem = new QTreeWidgetItem(this);
      groupItem->setText(NAME, QString("default"));
//      groupItem->setFlags(groupItem->flags() | Qt::ItemIsUserCheckable);
//      groupItem->setCheckState(VISIBILITY, Qt::Checked);
      groupItem->setText(ATOMCOUNT, QString::number(0));
      groupItem->setText(FORMULA, QString(""));
      groupItem->setText(OCCURRENCE, "0");
      groupItem->setData(0, Qt::UserRole, QVariant(-1));
    }

    // add molecule
    QTreeWidgetItem *molItem = new QTreeWidgetItem(groupItem);
    molItem->setText(NAME, QString((*iter)->getName().c_str()));
    molItem->setFlags(groupItem->flags() | Qt::ItemIsUserCheckable);
    molItem->setCheckState(VISIBILITY, Qt::Unchecked);
    molItem->setText(ATOMCOUNT, QString::number((*iter)->getAtomCount()));
    molItem->setText(FORMULA, QString((*iter)->getFormula().toString().c_str()));
    const int index = (*iter)->getId();
    molItem->setData(0, Qt::UserRole, QVariant(index));
    molItem->setSelected(World::getInstance().isSelected(*iter));

    // increase group occurrence
    int count = groupItem->text(4).toInt() + 1;
    groupItem->setText(OCCURRENCE, QString::number(count));
  }
  dirty = false;
  clearing = false;
}

void QtMoleculeList::paintEvent(QPaintEvent * event)
{
  if (dirty)
    refill();
  QTreeWidget::paintEvent(event);
}

void QtMoleculeList::subjectKilled(Observable *publisher) {
}

void QtMoleculeList::visibilityChanged(QTreeWidgetItem* item, int column)
{
  if (column == VISIBILITY) {
    const moleculeId_t molid = item->data(0, Qt::UserRole).toInt();
    const bool visible = item->checkState(VISIBILITY);
    emit moleculesVisibilityChanged(molid, visible);
  }
}

void QtMoleculeList::moleculeChanged() {
  /*int idx = verticalHeaderItem(row)->data(Qt::UserRole).toInt();
  molecule *mol = molecules->ReturnIndex(idx);
  string cellValue = item(row,NAME)->text().toStdString();
  if(mol->getName() != cellValue && cellValue !="") {
    mol->setName(cellValue);
  }
  else if(cellValue==""){
    item(row,NAME)->setText(QString(mol->getName().c_str()));
  }*/
}

void QtMoleculeList::rowsSelected(const QItemSelection & selected, const QItemSelection & deselected){

  if (clearing)
    return;
  if (selecting)
    return;
  selecting = true;

  // Select all molecules which belong to newly selected rows.
  QModelIndex index;
  QModelIndexList items = selected.indexes();
  foreach (index, items)
    if (index.column() == 0){
      int mol_id = model()->data(index, Qt::UserRole).toInt();
      if (mol_id < 0)
        continue;
      //std::cout << "select molecule" << std::endl;
      MoleCuilder::SelectionMoleculeById(mol_id);
    }

  // Unselect all molecules which belong to newly unselected rows.
  items = deselected.indexes();
  foreach (index, items)
    if (index.column() == 0){
      int mol_id = model()->data(index, Qt::UserRole).toInt();
      if (mol_id < 0)
        continue;
      //std::cout << "unselect molecule" << std::endl;
      MoleCuilder::SelectionNotMoleculeById(mol_id);
    }

  selecting = false;
}
