/*
 * AllAtomsInsideCuboidAction.cpp
 *
 *  Created on: Aug 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Actions/SelectionAction/AllAtomsInsideCuboidAction.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Descriptors/AtomDescriptor.hpp"
#include "Descriptors/AtomShapeDescriptor.hpp"
#include "atom.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"
#include "LinearAlgebra/Matrix.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "Shapes/BaseShapes.hpp"
#include "Shapes/Shape.hpp"
#include "Shapes/ShapeOps.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

using namespace std;

#include "UIElements/UIFactory.hpp"
#include "UIElements/Dialog.hpp"
#include "Actions/ValueStorage.hpp"


// memento to remember the state when undoing

class SelectionAllAtomsInsideCuboidState : public ActionState {
public:
  SelectionAllAtomsInsideCuboidState(std::vector<atom*> _selectedAtoms, const Vector &_position, const Vector &_extension, const double _Xangle, const double _Yangle, const double _Zangle) :
    selectedAtoms(_selectedAtoms),
    position(_position),
    extension(_extension),
    Xangle(_Xangle),
    Yangle(_Yangle),
    Zangle(_Zangle)
  {}
  std::vector<atom*> selectedAtoms;
  Vector position;
  Vector extension;
  double Xangle;
  double Yangle;
  double Zangle;
};

const char SelectionAllAtomsInsideCuboidAction::NAME[] = "select-atoms-inside-cuboid";

SelectionAllAtomsInsideCuboidAction::SelectionAllAtomsInsideCuboidAction() :
  Action(NAME)
{}

SelectionAllAtomsInsideCuboidAction::~SelectionAllAtomsInsideCuboidAction()
{}

void SelectionAllAtomsInsideCuboid(const Vector &position, const Vector &extension) {
  ValueStorage::getInstance().setCurrentValue(SelectionAllAtomsInsideCuboidAction::NAME, extension);
  ValueStorage::getInstance().setCurrentValue("position", position);
  ActionRegistry::getInstance().getActionByName(SelectionAllAtomsInsideCuboidAction::NAME)->call(Action::NonInteractive);
};

void SelectionAllAtomsInsideCuboid(const Vector &position, const Vector &extension, const double Xangle, const double Yangle, const double Zangle) {
  ValueStorage::getInstance().setCurrentValue(SelectionAllAtomsInsideCuboidAction::NAME, extension);
  ValueStorage::getInstance().setCurrentValue("position", position);
  ValueStorage::getInstance().setCurrentValue("angle-x", Xangle);
  ValueStorage::getInstance().setCurrentValue("angle-y", Yangle);
  ValueStorage::getInstance().setCurrentValue("angle-z", Zangle);
  ActionRegistry::getInstance().getActionByName(SelectionAllAtomsInsideCuboidAction::NAME)->call(Action::NonInteractive);
};

Dialog* SelectionAllAtomsInsideCuboidAction::fillDialog(Dialog *dialog) {
  ASSERT(dialog,"No Dialog given when filling action dialog");

  dialog->queryVector(NAME, false, ValueStorage::getInstance().getDescription(NAME));
  dialog->queryVector("position", false, ValueStorage::getInstance().getDescription("position"));

  return dialog;
}

Action::state_ptr SelectionAllAtomsInsideCuboidAction::performCall() {
  std::vector<atom *> selectedAtoms = World::getInstance().getSelectedAtoms();
  Vector position;
  Vector extension;
  double Xangle = 0.;
  double Yangle = 0.;
  double Zangle = 0.;
  Matrix RotationMatrix;

  ValueStorage::getInstance().queryCurrentValue("position", position);
  ValueStorage::getInstance().queryCurrentValue(NAME, extension);
  // check whether a rotation is given.
  if (ValueStorage::getInstance().queryCurrentValue("angle-x", Xangle, true))
    Xangle = 0.;
  if (ValueStorage::getInstance().queryCurrentValue("angle-y", Yangle, true))
    Yangle = 0.;
  if (ValueStorage::getInstance().queryCurrentValue("angle-z", Zangle, true))
    Zangle = 0.;
  RotationMatrix.rotation(Xangle, Yangle, Zangle);

  DoLog(1) && (Log() << Verbose(1) << "Selecting all atoms inside a rotated " << RotationMatrix << " cuboid at " << position << " and extension of " << extension << "." << endl);
  Shape s = translate(transform(stretch(Cuboid(),extension),RotationMatrix),position);
  World::getInstance().selectAllAtoms(AtomByShape(s));
  return Action::state_ptr(new SelectionAllAtomsInsideCuboidState(selectedAtoms, position, extension, Xangle, Yangle, Zangle));
}

Action::state_ptr SelectionAllAtomsInsideCuboidAction::performUndo(Action::state_ptr _state) {
  SelectionAllAtomsInsideCuboidState *state = assert_cast<SelectionAllAtomsInsideCuboidState*>(_state.get());

  World::getInstance().clearAtomSelection();
  for(std::vector<atom *>::iterator iter = state->selectedAtoms.begin(); iter != state->selectedAtoms.end(); ++iter)
    World::getInstance().selectAtom(*iter);

  return Action::state_ptr(_state);
}

Action::state_ptr SelectionAllAtomsInsideCuboidAction::performRedo(Action::state_ptr _state){
  SelectionAllAtomsInsideCuboidState *state = assert_cast<SelectionAllAtomsInsideCuboidState*>(_state.get());
  Matrix RotationMatrix;

  RotationMatrix.rotation(state->Xangle, state->Yangle, state->Zangle);
  Shape s = translate(transform(stretch(Cuboid(),state->extension),RotationMatrix),state->position);
  World::getInstance().selectAllAtoms(AtomByShape(s));

  return Action::state_ptr(_state);
}

bool SelectionAllAtomsInsideCuboidAction::canUndo() {
  return true;
}

bool SelectionAllAtomsInsideCuboidAction::shouldUndo() {
  return true;
}

const string SelectionAllAtomsInsideCuboidAction::getName() {
  return NAME;
}
