/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * IonUnitTest.cpp
 *
 *  Created on: Mar 10, 2014
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "CodePatterns/Assert.hpp"

#include <sstream>

#include "Element/ion.hpp"

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "IonUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( IonTest );


void IonTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  testelement = new element();
  testelement->setSymbol("H");
  testelement->Z = 1;
  testelement->Valence = +1.;
  testelement->NoValenceOrbitals = +1.;
  // create H+
  testion = new ion(*testelement, +1); 
};


void IonTest::tearDown()
{
  delete testelement;
  delete testion;
};

/** Test whether creations has correct asserts
 *
 */
void IonTest::CreationTest()
{
  ion *otherion = NULL;

  // check for zero ionization
#ifndef NDEBUG
  std::cout << "The following assertion is expected and does not mean a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW(otherion = new ion(testelement, 0), Assert::AssertionFailure);
#endif

  // check for ionization exceeding valence
#ifndef NDEBUG
  std::cout << "The following assertion is expected and does not mean a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW(otherion = new ion(testelement, +2), Assert::AssertionFailure);
#endif
}

/** Test whether assignment works
 *
 */
void IonTest::AssignmentTest()
{
  ion * otherion = new ion(testelement, -1);
  ion * const backup = otherion;
  CPPUNIT_ASSERT( otherion != testion );
  otherion = testion;
  CPPUNIT_ASSERT( otherion == testion );
  otherion = backup;
  *otherion = *testion;
  CPPUNIT_ASSERT( *otherion == *testion );
  delete backup;

  otherion = new ion(*testion);
  CPPUNIT_ASSERT( otherion != testion );
  CPPUNIT_ASSERT( *otherion == *testion );
  delete otherion;
}

/** Test whether the getters are working.
 *
 */
void IonTest::GetterTest()
{
  CPPUNIT_ASSERT_EQUAL( (atomId_t)0, testion->getAtomicNumber() );
  CPPUNIT_ASSERT_EQUAL( 0., testion->getValence() );
  CPPUNIT_ASSERT_EQUAL( +1., testion->getCharge() );

  ion * newion = new ion(testelement, -1);
  CPPUNIT_ASSERT_EQUAL( (atomId_t)2, newion->getAtomicNumber() );
  CPPUNIT_ASSERT_EQUAL( 2, newion->getNoValenceOrbitals() );
  CPPUNIT_ASSERT_EQUAL( -1., newion->getCharge() );
  delete newion;
}

/** Unit test for operator==()
 */
void IonTest::ComparisonTest()
{
  {
    ion * newion = new ion(testelement, -1);
    CPPUNIT_ASSERT( *testion != *newion );
    delete newion;
  }

  {
    ion * newion = new ion(testelement, +1);
    CPPUNIT_ASSERT( *testion == *newion );
    delete newion;
  }
}

/** Tests whether serialization of the class works
 */
void IonTest::SerializeTest()
{
  const std::string he("He");
  const std::string helium("Helium");
  testion->setSymbol(he);
  testion->setName(helium);

  CPPUNIT_ASSERT_EQUAL ( he, testion->getSymbol() );
  CPPUNIT_ASSERT_EQUAL( helium, testion->getName() );

  // write ion to stream
  std::stringstream stream;
  boost::archive::text_oarchive oa(stream);
  oa << testion;

  std::cout << "Contents of archive is " << stream.str() << std::endl;

  // create and open an archive for input
  boost::archive::text_iarchive ia(stream);
  // read class state from archive
  ion * newion = NULL;

  ia >> newion;

  CPPUNIT_ASSERT_EQUAL( he, newion->getSymbol() );
  CPPUNIT_ASSERT_EQUAL( helium, newion->getName() );

  CPPUNIT_ASSERT (*testion == *newion);

  delete newion;
};
