/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/** \file ion.cpp
 * 
 * Function implementations for the class ion.
 * 
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "ion.hpp"

/************************************* Functions for class ion **********************************/
ion::ion(const element &_element, const int _ionization) :
    element(_element),
    ionization(_ionization)
{
  // only outer shell can be ionized
  ASSERT( abs(ionization) <= NoValenceOrbitals,
      "ion::ion() - ionization ("+toString(ionization)+
      ") must not exceed valence ("+toString(NoValenceOrbitals)+").");
  ASSERT (ionization != 0,
     "ion::ion() - ionization must not be zero.");
}

ion::ion(const element* const _element, const int _ionization) :
    element(*_element),
    ionization(_ionization)
{
  // only outer shell can be ionized
  ASSERT( abs(ionization) <= NoValenceOrbitals,
      "ion::ion() - ionization ("+toString(ionization)+
      ") must not exceed valence ("+toString(NoValenceOrbitals)+").");
  ASSERT (ionization != 0,
     "ion::ion() - ionization must not be zero.");
}

ion::ion() :
    element(),
    ionization(0)
{

}

ion::~ion()
{}

ion& ion::operator=(const element &other)
{
  if (this != &other) { // check for self-assignment
    dynamic_cast<element &>(*this) = dynamic_cast<const element &>(other);
  }
  return *this;
}

ion& ion::operator=(const ion &other)
{
  if (this != &other) { // check for self-assignment
    dynamic_cast<element &>(*this) = dynamic_cast<const element &>(other);
    const_cast<int &>(ionization) = other.ionization;
  }
  return *this;
}

atomicNumber_t ion::getAtomicNumber() const
{
  return Z - ionization;
}

double ion::getCharge() const
{
  return (double)ionization;
}

double ion::getValence() const
{
  return Valence - ionization;
}

int ion::getNoValenceOrbitals() const
{
  return NoValenceOrbitals - ionization;
}

bool ion::operator==(const element &other) const {
  return dynamic_cast<const element &>(*this) == other;
}
bool ion::operator!=(const element &other) const {
  return !(*this == other);
}
bool ion::operator==(const ion &other) const {
  bool status = true;
  status &= dynamic_cast<const element &>(*this) == other;
  status &= ionization == other.ionization;
  return status;
}
