/*
 * QTDialog.cpp
 *
 *  Created on: Jan 18, 2010
 *      Author: crueger
 */

#include "UIElements/QT4/QTDialog.hpp"

#include <string>
#include <sstream>
#include <limits>

#include <Qt/qboxlayout.h>
#include <Qt/qlabel.h>
#include <Qt/qspinbox.h>
#include <QtGui/QDoubleSpinBox>
#include <Qt/qlineedit.h>
#include <Qt/qdialogbuttonbox.h>
#include <Qt/qpushbutton.h>
#include <Qt/qcombobox.h>

#include "World.hpp"
#include "periodentafel.hpp"
#include "atom.hpp"
#include "element.hpp"
#include "molecule.hpp"


using namespace std;

QTDialog::QTDialog() :
    QDialog(0)
{
  // creating and filling of the Dialog window
  mainLayout = new QVBoxLayout();
  inputLayout = new QVBoxLayout();
  buttonLayout = new QVBoxLayout();
  setLayout(mainLayout);
  mainLayout->addLayout(inputLayout);
  mainLayout->addLayout(buttonLayout);
  buttons = new QDialogButtonBox(QDialogButtonBox::Ok| QDialogButtonBox::Cancel);
  buttonLayout->addWidget(buttons);

  // Disable the ok button until something was entered
  buttons->button(QDialogButtonBox::Ok)->setEnabled(false);

  // connect the buttons to their appropriate slots
  connect(buttons, SIGNAL(accepted()), this, SLOT(accept()));
  connect(buttons, SIGNAL(rejected()), this, SLOT(reject()));
}

QTDialog::~QTDialog()
{
}

bool QTDialog::display(){
  // Button state might have changed by some update that
  // was done during query construction. To make sure
  // the state is correct, we just call update one more time.
  update();
  if(exec()) {
    setAll();
    return true;
  }
  else {
    return false;
  }
}

void QTDialog::update(){
  buttons->button(QDialogButtonBox::Ok)->setEnabled(checkAll());
}

/************************** Query Infrastructure ************************/

void QTDialog::queryInt(const char *title, int *target)
{
  registerQuery(new IntQTQuery(title,target,inputLayout,this));
}

void QTDialog::queryDouble(const char* title, double* target){
  registerQuery(new DoubleQTQuery(title,target,inputLayout,this));
}

void QTDialog::queryString(const char* title, std::string *target)
{
  registerQuery(new StringQTQuery(title,target,inputLayout,this));
}

void QTDialog::queryMolecule(const char *title,molecule **target,MoleculeListClass *molecules)
{
  registerQuery(new MoleculeQTQuery(title,target,molecules,inputLayout,this));
}

void QTDialog::queryVector(const char* title, Vector *target,const double *const cellSize, bool check) {
  registerQuery(new VectorQTQuery(title,target,cellSize,check,inputLayout,this));
}

void QTDialog::queryElement(const char* title, element **target){
  registerQuery(new ElementQTQuery(title,target,inputLayout,this));
}

/************************** Query Objects *******************************/

QTDialog::IntQTQuery::IntQTQuery(string _title,int *_target,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::IntQuery(_title,_target),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QSpinBox();
  inputBox->setValue(0);
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new IntQTQueryPipe(&tmp,_dialog);
  pipe->update(inputBox->value());
  connect(inputBox,SIGNAL(valueChanged(int)),pipe,SLOT(update(int)));
}

QTDialog::IntQTQuery::~IntQTQuery()
{
  delete pipe;
}

// Handling is easy since the GUI makes it impossible to enter invalid values
bool QTDialog::IntQTQuery::handle()
{
  return true;
}

QTDialog::DoubleQTQuery::DoubleQTQuery(string title,double *_target,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::DoubleQuery(title,_target),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QDoubleSpinBox();
  inputBox->setValue(0);
  inputBox->setRange(-numeric_limits<double>::max(),numeric_limits<double>::max());
  inputBox->setDecimals(3);
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new DoubleQTQueryPipe(&tmp,_dialog);
  pipe->update(inputBox->value());
  connect(inputBox,SIGNAL(valueChanged(double)),pipe,SLOT(update(double)));
}

QTDialog::DoubleQTQuery::~DoubleQTQuery()
{
  delete pipe;
}

bool QTDialog::DoubleQTQuery::handle() {
  return true;
}


QTDialog::StringQTQuery::StringQTQuery(string _title,string *_target,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::StringQuery(_title,_target),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QLineEdit();
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new StringQTQueryPipe(&tmp,_dialog);
  pipe->update(inputBox->text());
  connect(inputBox,SIGNAL(textChanged(const QString&)),pipe,SLOT(update(const QString&)));
}

QTDialog::StringQTQuery::~StringQTQuery()
{
  delete pipe;
}

// All values besides the empty string are valid
bool QTDialog::StringQTQuery::handle()
{
  return tmp!="";
}

QTDialog::MoleculeQTQuery::MoleculeQTQuery(string _title, molecule **_target, MoleculeListClass *_molecules, QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::MoleculeQuery(_title,_target,_molecules),
    parent(_parent)
{
  MoleculeList::iterator iter;
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  // add all molecules to the combo box
  for(iter  = molecules->ListOfMolecules.begin();
      iter != molecules->ListOfMolecules.end();
      ++iter) {
    stringstream sstr;
    sstr << (*iter)->IndexNr << "\t" << (*iter)->getName();
    inputBox->addItem(QString(sstr.str().c_str()),QVariant((*iter)->IndexNr));
  }
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new MoleculeQTQueryPipe(&tmp,_dialog,inputBox,_molecules);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QTDialog::MoleculeQTQuery::~MoleculeQTQuery()
{
  delete pipe;
}

// Handling is easy, since the GUI makes it impossible to select invalid values
bool QTDialog::MoleculeQTQuery::handle()
{
  return true;
}

QTDialog::VectorQTQuery::VectorQTQuery(std::string title, Vector *_target, const double *const _cellSize, bool _check,QBoxLayout *_parent,QTDialog *_dialog) :
    Dialog::VectorQuery(title,_target,_cellSize,_check),
    parent(_parent)
{
  // About the j: I don't know why it was done this way, but it was used this way in Vector::AskPosition, so I reused it
  int j = -1;
  const char *coords[3] = {"x:","y:","z:"};
  mainLayout= new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  mainLayout->addWidget(titleLabel);
  subLayout = new QVBoxLayout();
  mainLayout->addLayout(subLayout);
  for(int i=0; i<3; i++) {
    j+=i+1;
    coordLayout[i] = new QHBoxLayout();
    subLayout->addLayout(coordLayout[i]);
    coordLabel[i] = new QLabel(QString(coords[i]));
    coordLayout[i]->addWidget(coordLabel[i]);
    coordInput[i] = new QDoubleSpinBox();
    coordInput[i]->setRange(0,cellSize[j]);
    coordInput[i]->setDecimals(3);
    coordLayout[i]->addWidget(coordInput[i]);
    pipe[i] = new DoubleQTQueryPipe(&((*tmp)[i]),_dialog);
    pipe[i]->update(coordInput[i]->value());
    connect(coordInput[i],SIGNAL(valueChanged(double)),pipe[i],SLOT(update(double)));

  }
  parent->addLayout(mainLayout);
}

QTDialog::VectorQTQuery::~VectorQTQuery()
{}

bool QTDialog::VectorQTQuery::handle() {
  return true;
}


QTDialog::ElementQTQuery::ElementQTQuery(std::string _title, element **_target, QBoxLayout *_parent, QTDialog *_dialog) :
    Dialog::ElementQuery(_title,_target),
    parent(_parent)
{
  periodentafel *periode = World::get()->getPeriode();
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  element* Elemental = 0;
  for(Elemental = periode->start->next;
      Elemental!=periode->end;
      Elemental = Elemental->next)
  {
    stringstream sstr;
    sstr << Elemental->Z << "\t" << Elemental->name;
    inputBox->addItem(QString(sstr.str().c_str()),QVariant(Elemental->Z));
  }
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new ElementQTQueryPipe(&tmp,_dialog,inputBox);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QTDialog::ElementQTQuery::~ElementQTQuery()
{
  delete pipe;
}

bool QTDialog::ElementQTQuery::handle(){
  return true;
}

/*************************** Plumbing *******************************/

StringQTQueryPipe::StringQTQueryPipe(string *_content, QTDialog *_dialog) :
  content(_content),
  dialog(_dialog)
{}

StringQTQueryPipe::~StringQTQueryPipe()
{}

void StringQTQueryPipe::update(const QString& newText) {
  content->assign(newText.toStdString());
  dialog->update();
}

IntQTQueryPipe::IntQTQueryPipe(int *_content, QTDialog *_dialog) :
  content(_content),
  dialog(_dialog)
{}

IntQTQueryPipe::~IntQTQueryPipe()
{}

void IntQTQueryPipe::update(int newInt) {
  (*content) = newInt;
  dialog->update();
}

DoubleQTQueryPipe::DoubleQTQueryPipe(double *_content, QTDialog *_dialog) :
  content(_content),
  dialog(_dialog)
{}

DoubleQTQueryPipe::~DoubleQTQueryPipe()
{}

void DoubleQTQueryPipe::update(double newDbl) {
  (*content) = newDbl;
  dialog->update();
}

MoleculeQTQueryPipe::MoleculeQTQueryPipe(molecule **_content, QTDialog *_dialog, QComboBox *_theBox, MoleculeListClass *_molecules) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox),
  molecules(_molecules)
{}

MoleculeQTQueryPipe::~MoleculeQTQueryPipe()
{}

void MoleculeQTQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  (*content) = molecules->ReturnIndex(idx);
  dialog->update();
}

ElementQTQueryPipe::ElementQTQueryPipe(element **_content, QTDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

ElementQTQueryPipe::~ElementQTQueryPipe()
{}

void ElementQTQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  (*content) = World::get()->getPeriode()->FindElement(idx);
  dialog->update();
}

