/*
 * MoleculeDescriptorTest.cpp
 *
 *  Created on: Mar 4, 2010
 *      Author: crueger
 */

#include "MoleculeDescriptorTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>
#include <iostream>

#include <Descriptors/MoleculeDescriptor.hpp>
#include <Descriptors/MoleculeIdDescriptor.hpp>

#include "World.hpp"
#include "molecule.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/
// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( MoleculeDescriptorTest );

// set up and tear down
void MoleculeDescriptorTest::setUp(){
  World::getInstance();
  for(int i=0;i<MOLECULE_COUNT;++i){
    molecules[i]= World::getInstance().createMolecule();
    moleculeIds[i]= molecules[i]->getId();
  }
}

void MoleculeDescriptorTest::tearDown(){
  World::purgeInstance();
}

// some helper functions
static bool hasAllMolecules(std::vector<molecule*> molecules,moleculeId_t ids[MOLECULE_COUNT], std::set<moleculeId_t> excluded = std::set<moleculeId_t>()){
  for(int i=0;i<MOLECULE_COUNT;++i){
    moleculeId_t id = ids[i];
    if(!excluded.count(id)){
      std::vector<molecule*>::iterator iter;
      bool res=false;
      for(iter=molecules.begin();iter!=molecules.end();++iter){
        res |= (*iter)->getId() == id;
      }
      if(!res) {
        cout << "Molecule " << id << " missing in returned list" << endl;
        return false;
      }
    }
  }
  return true;
}

static bool hasNoDuplicateMolecules(std::vector<molecule*> molecules){
  std::set<moleculeId_t> found;
  std::vector<molecule*>::iterator iter;
  for(iter=molecules.begin();iter!=molecules.end();++iter){
    int id = (*iter)->getId();
    if(found.count(id))
      return false;
    found.insert(id);
  }
  return true;
}


void MoleculeDescriptorTest::MoleculeBaseSetsTest(){
  std::vector<molecule*> allMolecules = World::getInstance().getAllMolecules(AllMolecules());
  CPPUNIT_ASSERT_EQUAL( true , hasAllMolecules(allMolecules,moleculeIds));
  CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicateMolecules(allMolecules));

  std::vector<molecule*> noMolecules = World::getInstance().getAllMolecules(NoMolecules());
  CPPUNIT_ASSERT_EQUAL( true , noMolecules.empty());
}
void MoleculeDescriptorTest::MoleculeIdTest(){
  // test Molecules from boundaries and middle of the set
  molecule* testMolecule;
  testMolecule = World::getInstance().getMolecule(MoleculeById(moleculeIds[0]));
  CPPUNIT_ASSERT(testMolecule);
  CPPUNIT_ASSERT_EQUAL( moleculeIds[0], testMolecule->getId());
  testMolecule = World::getInstance().getMolecule(MoleculeById(moleculeIds[MOLECULE_COUNT/2]));
  CPPUNIT_ASSERT(testMolecule);
  CPPUNIT_ASSERT_EQUAL( moleculeIds[MOLECULE_COUNT/2], testMolecule->getId());
  testMolecule = World::getInstance().getMolecule(MoleculeById(moleculeIds[MOLECULE_COUNT-1]));
  CPPUNIT_ASSERT(testMolecule);
  CPPUNIT_ASSERT_EQUAL( moleculeIds[MOLECULE_COUNT-1], testMolecule->getId());

  // find some ID that has not been created
  moleculeId_t outsideId=0;
  bool res = false;
  for(outsideId=0;!res;++outsideId) {
    res = true;
    for(int i = 0; i < MOLECULE_COUNT; ++i){
      res &= moleculeIds[i]!=outsideId;
    }
  }
  // test from outside of set
  testMolecule = World::getInstance().getMolecule(MoleculeById(outsideId));
  CPPUNIT_ASSERT(!testMolecule);
}
void MoleculeDescriptorTest::MoleculeCalcTest(){
  // test some elementary set operations
  {
    std::vector<molecule*> testMolecules = World::getInstance().getAllMolecules(AllMolecules()||NoMolecules());
    CPPUNIT_ASSERT_EQUAL( true , hasAllMolecules(testMolecules,moleculeIds));
    CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicateMolecules(testMolecules));
  }

  {
    std::vector<molecule*> testMolecules = World::getInstance().getAllMolecules(NoMolecules()||AllMolecules());
    CPPUNIT_ASSERT_EQUAL( true , hasAllMolecules(testMolecules,moleculeIds));
    CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicateMolecules(testMolecules));
  }

  {
    std::vector<molecule*> testMolecules = World::getInstance().getAllMolecules(NoMolecules()&&AllMolecules());
    CPPUNIT_ASSERT_EQUAL( true , testMolecules.empty());
  }

  {
    std::vector<molecule*> testMolecules = World::getInstance().getAllMolecules(AllMolecules()&&NoMolecules());
    CPPUNIT_ASSERT_EQUAL( true , testMolecules.empty());
  }

  {
    std::vector<molecule*> testMolecules = World::getInstance().getAllMolecules(!AllMolecules());
    CPPUNIT_ASSERT_EQUAL( true , testMolecules.empty());
  }

  {
    std::vector<molecule*> testMolecules = World::getInstance().getAllMolecules(!NoMolecules());
    CPPUNIT_ASSERT_EQUAL( true , hasAllMolecules(testMolecules,moleculeIds));
    CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicateMolecules(testMolecules));
  }

  // exclude and include some molecules
  {
    std::vector<molecule*> testMolecules = World::getInstance().getAllMolecules(AllMolecules()&&(!MoleculeById(moleculeIds[MOLECULE_COUNT/2])));
    std::set<moleculeId_t> excluded;
    excluded.insert(moleculeIds[MOLECULE_COUNT/2]);
    CPPUNIT_ASSERT_EQUAL( true , hasAllMolecules(testMolecules,moleculeIds,excluded));
    CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicateMolecules(testMolecules));
    CPPUNIT_ASSERT_EQUAL( (size_t)(MOLECULE_COUNT-1), testMolecules.size());
  }

  {
    std::vector<molecule*> testMolecules = World::getInstance().getAllMolecules(NoMolecules()||(MoleculeById(moleculeIds[MOLECULE_COUNT/2])));
    CPPUNIT_ASSERT_EQUAL( (size_t)1, testMolecules.size());
    CPPUNIT_ASSERT_EQUAL( moleculeIds[MOLECULE_COUNT/2], testMolecules[0]->getId());
  }
}
