/*
 * RotateAroundSelfByAngleAction.cpp
 *
 *  Created on: Aug 06, 2010
 *      Author: heber
 */

#include "Helpers/MemDebug.hpp"

#include "Actions/MoleculeAction/RotateAroundSelfByAngleAction.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"
#include "LinearAlgebra/Line.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "atom.hpp"
#include "molecule.hpp"


#include <iostream>
#include <fstream>
#include <string>

using namespace std;

#include "UIElements/UIFactory.hpp"
#include "UIElements/Dialog.hpp"
#include "Actions/ValueStorage.hpp"

/****** MoleculeRotateAroundSelfByAngleAction *****/

// memento to remember the state when undoing

class MoleculeRotateAroundSelfByAngleState : public ActionState {
public:
    MoleculeRotateAroundSelfByAngleState(molecule * const _mol, const double _alpha) :
      mol(_mol),
      alpha(_alpha)
    {}
    molecule* const mol;
    double alpha;
};

const char MoleculeRotateAroundSelfByAngleAction::NAME[] = "rotate-self";

MoleculeRotateAroundSelfByAngleAction::MoleculeRotateAroundSelfByAngleAction() :
  Action(NAME)
{}

MoleculeRotateAroundSelfByAngleAction::~MoleculeRotateAroundSelfByAngleAction()
{}

void MoleculeRotateAroundSelfByAngle(double angle) {
  ValueStorage::getInstance().setCurrentValue(MoleculeRotateAroundSelfByAngleAction::NAME, angle);
  ActionRegistry::getInstance().getActionByName(MoleculeRotateAroundSelfByAngleAction::NAME)->call(Action::NonInteractive);
};

Dialog* MoleculeRotateAroundSelfByAngleAction::fillDialog(Dialog *dialog) {
  ASSERT(dialog,"No Dialog given when filling action dialog");

  dialog->queryDouble(NAME, MapOfActions::getInstance().getDescription(NAME));

  return dialog;
}

Action::state_ptr MoleculeRotateAroundSelfByAngleAction::performCall() {
  molecule *mol = NULL;
  double alpha = 0.;

  // obtain angle around which rotate
  ValueStorage::getInstance().queryCurrentValue(NAME, alpha);

  // check whether a single atom and molecule is selected
  if ((World::getInstance().getSelectedAtoms().size() != 1) || (World::getInstance().getSelectedMolecules().size() != 1))
    return Action::failure;

  DoLog(0) && (Log() << Verbose(0) << "Rotate around self by " << alpha << "." << endl);
  mol = World::getInstance().beginMoleculeSelection()->second;

  // Creation Line that is the rotation axis
  Vector *CenterOfGravity = mol->DetermineCenterOfGravity();
  Line RotationAxis(*CenterOfGravity, (World::getInstance().beginAtomSelection()->second)->x);
  delete(CenterOfGravity);

  for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
    *((*iter)->node) = RotationAxis.rotateVector(*((*iter)->node), alpha);
  }
  DoLog(0) && (Log() << Verbose(0) << "done." << endl);

  return Action::state_ptr(new MoleculeRotateAroundSelfByAngleState(mol, alpha));
}

Action::state_ptr MoleculeRotateAroundSelfByAngleAction::performUndo(Action::state_ptr _state) {
  MoleculeRotateAroundSelfByAngleState *state = assert_cast<MoleculeRotateAroundSelfByAngleState*>(_state.get());

  Vector *CenterOfGravity = state->mol->DetermineCenterOfGravity();
  Line RotationAxis(*CenterOfGravity, (World::getInstance().beginAtomSelection()->second)->x);
  delete(CenterOfGravity);

  for (molecule::iterator iter = state->mol->begin(); iter != state->mol->end(); ++iter) {
    *((*iter)->node) = RotationAxis.rotateVector(*((*iter)->node), -state->alpha);
  }

  return Action::state_ptr(_state);
}

Action::state_ptr MoleculeRotateAroundSelfByAngleAction::performRedo(Action::state_ptr _state){
  MoleculeRotateAroundSelfByAngleState *state = assert_cast<MoleculeRotateAroundSelfByAngleState*>(_state.get());

  Vector *CenterOfGravity = state->mol->DetermineCenterOfGravity();
  Line RotationAxis(*CenterOfGravity, (World::getInstance().beginAtomSelection()->second)->x);
  delete(CenterOfGravity);

  for (molecule::iterator iter = state->mol->begin(); iter != state->mol->end(); ++iter) {
    *((*iter)->node) = RotationAxis.rotateVector(*((*iter)->node), state->alpha);
  }

  return Action::state_ptr(_state);
}

bool MoleculeRotateAroundSelfByAngleAction::canUndo() {
  return true;
}

bool MoleculeRotateAroundSelfByAngleAction::shouldUndo() {
  return true;
}

const string MoleculeRotateAroundSelfByAngleAction::getName() {
  return NAME;
}
