/*
 * RotateAroundOriginByAngleAction.cpp
 *
 *  Created on: Aug 06, 2010
 *      Author: heber
 */

#include "Helpers/MemDebug.hpp"

#include "Actions/MoleculeAction/RotateAroundOriginByAngleAction.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"
#include "LinearAlgebra/Line.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "molecule.hpp"


#include <iostream>
#include <fstream>
#include <string>

using namespace std;

#include "UIElements/UIFactory.hpp"
#include "UIElements/Dialog.hpp"
#include "Actions/ValueStorage.hpp"

/****** MoleculeRotateAroundOriginByAngleAction *****/

// memento to remember the state when undoing

class MoleculeRotateAroundOriginByAngleState : public ActionState {
public:
  MoleculeRotateAroundOriginByAngleState(const std::vector<molecule*> &_molecules,const Vector &_Axis, const double _alpha) :
    molecules(_molecules),
    Axis(_Axis),
    alpha(_alpha)
  {}
  std::vector<molecule*> molecules;
  Vector Axis;
  double alpha;
};

const char MoleculeRotateAroundOriginByAngleAction::NAME[] = "rotate-origin";

MoleculeRotateAroundOriginByAngleAction::MoleculeRotateAroundOriginByAngleAction() :
  Action(NAME)
{}

MoleculeRotateAroundOriginByAngleAction::~MoleculeRotateAroundOriginByAngleAction()
{}

void MoleculeRotateAroundOriginByAngle(double angle) {
  ValueStorage::getInstance().setCurrentValue(MoleculeRotateAroundOriginByAngleAction::NAME, angle);
  ActionRegistry::getInstance().getActionByName(MoleculeRotateAroundOriginByAngleAction::NAME)->call(Action::NonInteractive);
};

Dialog* MoleculeRotateAroundOriginByAngleAction::fillDialog(Dialog *dialog) {
  ASSERT(dialog,"No Dialog given when filling action dialog");

  dialog->queryDouble(NAME, MapOfActions::getInstance().getDescription(NAME));
  dialog->queryVector("position", false, MapOfActions::getInstance().getDescription("position"));

  return dialog;
}

Action::state_ptr MoleculeRotateAroundOriginByAngleAction::performCall() {
  molecule *mol = NULL;
  double alpha = 0.;
  Vector Axis;

  // obtain axis to rotate to
  ValueStorage::getInstance().queryCurrentValue(NAME, alpha);
  ValueStorage::getInstance().queryCurrentValue("position", Axis);

  // check whether Axis is valid
  if (Axis.IsZero())
    return Action::failure;

  // convert from degrees to radian
  alpha *= M_PI/180.;

  DoLog(0) && (Log() << Verbose(0) << "Rotate around origin by " << alpha << " radian, axis from origin to " << Axis << "." << endl);
  for (World::MoleculeSelectionIterator iter = World::getInstance().beginMoleculeSelection(); iter != World::getInstance().endMoleculeSelection(); ++iter) {
    mol = iter->second;

    // Creation Line that is the rotation axis
    Line RotationAxis(Vector(0.,0.,0.), Axis);

    for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
      *((*iter)->node) = RotationAxis.rotateVector(*((*iter)->node), alpha);
    }
    DoLog(0) && (Log() << Verbose(0) << "done." << endl);
  }
  return Action::state_ptr(new MoleculeRotateAroundOriginByAngleState(World::getInstance().getSelectedMolecules(), Axis, alpha));
}

Action::state_ptr MoleculeRotateAroundOriginByAngleAction::performUndo(Action::state_ptr _state) {
  MoleculeRotateAroundOriginByAngleState *state = assert_cast<MoleculeRotateAroundOriginByAngleState*>(_state.get());
  molecule *mol = NULL;

  for (std::vector<molecule *>::const_iterator iter = state->molecules.begin(); iter != state->molecules.end(); ++iter) {
    mol = *iter;

    // Creation Line that is the rotation axis
    Line RotationAxis(Vector(0.,0.,0.), state->Axis);

    for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
      *((*iter)->node) = RotationAxis.rotateVector(*((*iter)->node), -state->alpha);
    }
  }

  return Action::state_ptr(_state);
}

Action::state_ptr MoleculeRotateAroundOriginByAngleAction::performRedo(Action::state_ptr _state){
  MoleculeRotateAroundOriginByAngleState *state = assert_cast<MoleculeRotateAroundOriginByAngleState*>(_state.get());
  molecule *mol = NULL;

  for (std::vector<molecule *>::const_iterator iter = state->molecules.begin(); iter != state->molecules.end(); ++iter) {
    mol = *iter;

    // Creation Line that is the rotation axis
    Line RotationAxis(Vector(0.,0.,0.), state->Axis);

    for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
      *((*iter)->node) = RotationAxis.rotateVector(*((*iter)->node), state->alpha);
    }
  }

  return Action::state_ptr(_state);
}

bool MoleculeRotateAroundOriginByAngleAction::canUndo() {
  return true;
}

bool MoleculeRotateAroundOriginByAngleAction::shouldUndo() {
  return true;
}

const string MoleculeRotateAroundOriginByAngleAction::getName() {
  return NAME;
}
